// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.terminal.kernel8;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;
import java.io.File;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Random;

import javax.swing.*;
import javax.swing.LayoutStyle.ComponentPlacement;

import org.denom.*;
import org.denom.format.BerTLV;
import org.denom.format.BerTLVList;
import org.denom.format.JSONObject;
import org.denom.log.Colors;
import org.denom.smartcard.emv.TagEmv;
import org.denom.smartcard.emv.kernel8.*;
import org.denom.smartcard.emv.kernel8.struct.*;
import org.denom.swing.SwingUtils;

import static org.denom.Ex.*;
import static org.denom.Binary.Bin;

/**
 * Panel with transaction params.
 */
@SuppressWarnings("serial")
class PanelTransaction extends JPanel
{
	TerminalKernel8GUI main;

	JTextField textTrcAmount;
	JTextField textAID;
	JTextField textTerminalType;
	JTextField textTransactionType;
	JTextField textTrcCurrencyCode;
	JTextField textReadDataTags;
	JTextArea textWriteDataTags;
	JCheckBox checkDataStorage;

	// -----------------------------------------------------------------------------------------------------------------
	PanelTransaction( TerminalKernel8GUI main )
	{
		this.main = main;
		setBorder( BorderFactory.createEmptyBorder( 10, 10, 10, 10 ) );

		JLabel labelInfo = new JLabel( "<html>Performing transaction with card application.</html>" );
		Color labelColor = new Color( 50, 50, 155 );
		labelInfo.setForeground( labelColor );

		JLabel labelAID = new JLabel( "Application AID" );
		textAID = SwingUtils.CreateLimitedTextField( 50 );
		textAID.setToolTipText( "Hex, 5-16 bytes" );

		JLabel labelTerminalType = new JLabel( "Terminal Type" );
		textTerminalType = SwingUtils.CreateLimitedTextField( 2 );
		textTerminalType.setToolTipText( "Tag 9F35. Format: n 2. Example: 12" );

		JLabel labelTransactionType = new JLabel( "Transaction Type" );
		textTransactionType = SwingUtils.CreateLimitedTextField( 2 );
		textTransactionType.setToolTipText( "Tag 9C. Format: n 2. First two digits of the ISO 8583:1987 Processing Code" );

		JLabel labelTrcAmount = new JLabel( "Amount" );
		textTrcAmount = SwingUtils.CreateLimitedTextField( 15 );
		textTrcAmount.setToolTipText( "Tag 9F02. Amount must be 'n 12' (up to 12 numbers)" );

		JLabel labelTrcCurrencyCode = new JLabel( "Currency Code" );
		textTrcCurrencyCode = SwingUtils.CreateLimitedTextField( 3 );
		textTrcCurrencyCode.setToolTipText( "Tag 9F3C. Format: n 3 (3 numbers). Example: 643" );

		JLabel labelDS = new JLabel( "<html>Data Storage:" );
		labelDS.setForeground( labelColor );
		checkDataStorage = new JCheckBox( "Data Storage:" );
		checkDataStorage.addItemListener( this::onCheckBoxDataStorage );
		
		JLabel labelReadDataTags = new JLabel( "Read Data" );
		textReadDataTags = SwingUtils.CreateLimitedTextField( 150 );
		textReadDataTags.setToolTipText( "Tag List for commands Read Data ('9F8111'-'9F811A'). Example: 9F8111 9F8112" );

		JLabel labelWriteDataTags = new JLabel( "Write Data" );
		textWriteDataTags = new JTextArea();
		textWriteDataTags.setToolTipText( "TLV List for commands Write Data (TLVs with tags '9F8111'-'9F811A')" );
		textWriteDataTags.setDocument( SwingUtils.createLimitedSizeDocument( 4000 ) );
		textWriteDataTags.setLineWrap( true );
		JScrollPane scrollPaneWriteData = new JScrollPane( textWriteDataTags );

		JButton buttonTransaction = new JButton( "Perform transaction" );
		buttonTransaction.addActionListener( this::onTransaction );

		int prefSize = GroupLayout.PREFERRED_SIZE;
		int compHeight = 30;
		GroupLayout layout = new GroupLayout( this );
		this.setLayout( layout );

		layout.setHorizontalGroup( layout.createParallelGroup()
			.addComponent( labelInfo )

			.addGroup( layout.createSequentialGroup()
				.addComponent( labelAID, prefSize, 150, prefSize )
				.addComponent( textAID, prefSize, 250, prefSize ) )

			.addGroup( layout.createSequentialGroup()
					.addComponent( labelTerminalType, prefSize, 150, prefSize )
					.addComponent( textTerminalType, prefSize, 50, prefSize ) )

			.addGroup( layout.createSequentialGroup()
					.addComponent( labelTransactionType, prefSize, 150, prefSize )
					.addComponent( textTransactionType, prefSize, 50, prefSize ) )

			.addGroup( layout.createSequentialGroup()
					.addComponent( labelTrcCurrencyCode, prefSize, 150, prefSize )
					.addComponent( textTrcCurrencyCode, prefSize, 50, prefSize ) )

			.addGroup( layout.createSequentialGroup()
				.addComponent( labelTrcAmount, prefSize, 150, prefSize )
				.addComponent( textTrcAmount, prefSize, 150, prefSize ) )

			.addComponent( checkDataStorage )
			
			.addGroup( layout.createSequentialGroup()
					.addComponent( labelReadDataTags, prefSize, 100, prefSize )
					.addComponent( textReadDataTags, 100, 300, 1200 ) )

			.addGroup( layout.createSequentialGroup()
					.addComponent( labelWriteDataTags, prefSize, 100, prefSize )
					.addComponent( scrollPaneWriteData, 100, 300, 1200 ) )

			.addGroup( layout.createSequentialGroup()
					.addPreferredGap( ComponentPlacement.UNRELATED, 5, Short.MAX_VALUE )
					.addComponent( buttonTransaction, prefSize, 250, prefSize ) )
		);

		layout.setVerticalGroup( layout.createSequentialGroup()
			.addComponent( labelInfo )
			.addGap( 25 )

			.addGroup( layout.createParallelGroup()
				.addComponent( labelAID, prefSize, compHeight, prefSize )
				.addComponent( textAID, prefSize, compHeight, prefSize ) )
			.addGap( 15 )

			.addGroup( layout.createParallelGroup()
					.addComponent( labelTerminalType, prefSize, compHeight, prefSize )
					.addComponent( textTerminalType, prefSize, compHeight, prefSize ) )
			.addGap( 15 )

			.addGroup( layout.createParallelGroup()
					.addComponent( labelTransactionType, prefSize, compHeight, prefSize )
					.addComponent( textTransactionType, prefSize, compHeight, prefSize ) )

			.addGroup( layout.createParallelGroup()
					.addComponent( labelTrcCurrencyCode, prefSize, compHeight, prefSize )
					.addComponent( textTrcCurrencyCode, prefSize, compHeight, prefSize ) )

			.addGroup( layout.createParallelGroup()
					.addComponent( labelTrcAmount, prefSize, compHeight, prefSize )
					.addComponent( textTrcAmount, prefSize, compHeight, prefSize ) )

			.addGap( 15 )
			.addComponent( checkDataStorage )
			.addGap( 10 )

			.addGroup( layout.createParallelGroup()
					.addComponent( labelReadDataTags, prefSize, compHeight, prefSize )
					.addComponent( textReadDataTags, prefSize, compHeight, prefSize ) )
			
			.addGroup( layout.createParallelGroup()
					.addComponent( labelWriteDataTags, prefSize, compHeight, prefSize )
					.addComponent( scrollPaneWriteData, prefSize, 80, prefSize ) )

			.addGap( 20 )
			.addComponent( buttonTransaction, prefSize, 50, prefSize )
		);
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void setEnabledDSComponents( boolean isEnabled )
	{
		textReadDataTags.setEnabled( isEnabled );
		textWriteDataTags.setEnabled( isEnabled );
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void onCheckBoxDataStorage( ItemEvent e )
	{
		// Проверка состояния галочки
		if( e.getStateChange() == ItemEvent.SELECTED )
			setEnabledDSComponents( true );
		if( e.getStateChange() == ItemEvent.DESELECTED )
			setEnabledDSComponents( false );
	}

	// -----------------------------------------------------------------------------------------------------------------
	private TlvDatabase initSessParams()
	{
		TlvDatabase sessParams = new TlvDatabase( new TagDictKernel8() );

		// -------------------------------------------------------------------
		// AID
		Binary aid = Bin( textAID.getText() );
		MUST( (aid.size() >= 5) && (aid.size() <= 16), "Wrong AID Len" );
		sessParams.store( TagEmv.ApplicationIdentifierTerminal, aid );

		// -------------------------------------------------------------------
		// Terminal Type
		String s = textTerminalType.getText();
		MUST( (s.length() == 2) && s.matches( "[0-9]+" ), "'Terminal type' must be n 2 (only numbers)" );
		sessParams.store( TagEmv.TerminalType, Bin( s ) );

		// -------------------------------------------------------------------
		// Transaction Type
		s = textTransactionType.getText();
		MUST( (s.length() == 2) && s.matches( "[0-9]+" ), "'Transaction type' must be n 2 (only numbers)" );
		sessParams.store( TagEmv.TransactionType, Bin( s ) );

		// -------------------------------------------------------------------
		// Transaction Currency Code
		s = textTrcCurrencyCode.getText();
		MUST( (s.length() == 3) && s.matches( "[0-9]+" ), "'Transaction currency code' must be n 3 (only numbers)" );
		s = "0" + s;
		sessParams.store( TagEmv.TransactionCurrencyCode, Bin( s ) );

		// -------------------------------------------------------------------
		// Transaction Currency Exponent
		sessParams.store( TagEmv.TransactionCurrencyExponent, Bin("02") );

		// -------------------------------------------------------------------
		// Сумма транзакции - Amount Authorised (Numeric)
		s = textTrcAmount.getText();
		MUST( (s.length() <= 12) && s.matches( "[0-9]+" ), "Amount must be n 12 (only numbers)" );
		s = Strings.PadLeft( s, 12, '0' );
		sessParams.store( TagEmv.AmountAuthorisedNumeric, Bin( s ) );

		// -------------------------------------------------------------------
		// Amount Other
		sessParams.store( TagEmv.AmountOtherNumeric,      Bin("000000000000") );

		// -------------------------------------------------------------------
		// Transaction Date
		Binary nowDate = Bin( ZonedDateTime.now( ZoneOffset.UTC ).format( DateTimeFormatter.ofPattern("yyMMdd") ) );
		sessParams.store( TagEmv.TransactionDate, nowDate );
		// Transaction Time
		Binary nowTime = Bin( ZonedDateTime.now( ZoneOffset.UTC ).format( DateTimeFormatter.ofPattern("HHmmss") ) );
		sessParams.store( TagEmv.TransactionTime, nowTime );

		return sessParams;
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void onTransaction( ActionEvent e )
	{
		main.runScript( "Perform Transaction", () ->
		{
			MUST( new File( main.CONFIG_KERNEL_FILENAME ).exists(), "File not found: " + main.CONFIG_KERNEL_FILENAME );
			JSONObject joKernelConfig = new JSONObject().loadWithComments( main.CONFIG_KERNEL_FILENAME );

			ArrayList<IKernel8Crypter> crypters = new ArrayList<>();
			crypters.add( new Kernel8CrypterNIST() );
			TerminalKernel8 term = new TerminalKernel8( joKernelConfig, main.cr, main.log, new Random( System.nanoTime() ), crypters );
			TlvDatabase sessParams = initSessParams();

			if( checkDataStorage.isSelected() )
			{
				Binary bin = Bin( textWriteDataTags.getText() );
				MUST( BerTLV.isTLVList( bin ), "Wrong data for Write Data" );
				BerTLVList tlvList = new BerTLVList( bin );
				for( BerTLV tlv : tlvList.recs )
					MUST( (tlv.tag >= TagKernel8.DataEnvelopeMin) && (tlv.tag <= TagKernel8.DataEnvelopeMax), "Wrong tag for Write Data" );
				sessParams.store( TagKernel8.DataEnvelopesToWrite, bin );
			}

			Ticker t = new Ticker();

			OUT out = term.startTransaction( sessParams );
			if( out != null )
			{
				main.log.writeln( Colors.RED_I, "OUT:" );
				main.log.writeln( Colors.RED_I, new BerTLVList( out.outData ).toString( 0 ) );
				THROW( "Transaction aborted" );
			}

			if( checkDataStorage.isSelected() )
			{
				// Считать из DS
				String s = textReadDataTags.getText();
				Binary tags = Bin( s );
				Arr<Integer> arr = BerTLV.parseTagList( tags );
				for( int tag : arr )
				{
					MUST( (tag >= TagKernel8.DataEnvelopeMin) && (tag <= TagKernel8.DataEnvelopeMax), "Wrong tag for Read Data" );
					Binary tagData = term.readData( tag );
					if( tagData != null )
					{
						main.log.writeln( tagData.Hex( 1 ) );
					}
					else
					{
						main.log.writeln( Colors.YELLOW_I, "Can't read tag " + Binary.Num_Bin( tag, 3 ).Hex() );
					}
				}
			}

			out = term.generateAC();

			BerTLVList outTlvs = new BerTLVList( out.outData );
			main.log.writeln( Colors.GREEN, "OUT:" );
			main.log.writeln( Colors.GREEN, sessParams.dict.BerTLVListToString( outTlvs, 0 ) );

			Binary uird1 = outTlvs.find( TagKernel8.UserInterfaceRequestData1 ).value;
			if( !uird1.empty() )
				main.log.writeln( Colors.GREEN_I, "MSG: " + MessageIdentifier.getTextEngl( uird1.get( 0 ) ) );

			main.log.writeln( Colors.MAGENTA_I, "Transaction Time: " + t.getDiffMs() + " ms" );
		} );
	}

	// -----------------------------------------------------------------------------------------------------------------
	 JSONObject toJSON()
	{
		JSONObject jo = new JSONObject();

		jo.put( "Application AID", textAID.getText() );
		jo.put( "Transaction Amount", textTrcAmount.getText() );
		jo.put( "Terminal Type", textTerminalType.getText() );
		jo.put( "Transaction Type", textTransactionType.getText() );
		jo.put( "Transaction Currency Code", textTrcCurrencyCode.getText() );
		jo.put( "Read Data Tags", textReadDataTags.getText() );
		jo.put( "Write Data Tags", textWriteDataTags.getText() );
		jo.put( "Use Data Storage", checkDataStorage.isSelected() );

		return jo;
	}

	// -----------------------------------------------------------------------------------------------------------------
	void fromJSON( JSONObject jo )
	{
		textAID.setText( jo.optString( "Application AID" ) );
		textTrcAmount.setText( jo.optString( "Transaction Amount" ) );
		textTerminalType.setText( jo.optString( "Terminal Type" ) );
		textTransactionType.setText( jo.optString( "Transaction Type" ) );
		textTrcCurrencyCode.setText( jo.optString( "Transaction Currency Code" ) );
		textReadDataTags.setText( jo.optString( "Read Data Tags" ) );
		textWriteDataTags.setText( jo.optString( "Write Data Tags" ) );

		checkDataStorage.setSelected( jo.optBoolean( "Use Data Storage", false ) );
		setEnabledDSComponents( checkDataStorage.isSelected() );
	}

}

