// Denom.org
// bouncycastle.org

package org.denom.crypt.blockcipher;

import org.denom.Binary;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

import static org.denom.crypt.blockcipher.CAST5.*;

/**
 * CAST6 key encryption operations, such as encoding data and generating keys.
 * RFC2612 - CAST6 (128bit block, 128-256bit key).
 */
public final class CAST6 extends BlockCipher
{
	protected static final int BLOCK_SIZE = 16;  // bytes = 128 bits

	/**
	 * Put the round and mask keys into an array.
	 * Kr0[i] => _Kr[i*4 + 0]
	 */
	private int[] Kr = new int[ 12 * 4 ]; // the rotating round key(s)
	private int[] Km = new int[ 12 * 4 ]; // the masking round key(s)

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public CAST6()
	{
		this( Bin( 16 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public CAST6( final Binary key )
	{
		super.initialize( BLOCK_SIZE );
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public CAST6 clone()
	{
		return new CAST6( this.key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "CAST6";
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key - 16 <= keySize <= 32.
	 */
	@Override
	public void setKey( final Binary key )
	{
		MUST( (key.size() >= 16) && (key.size() <= 32), "Invalid key size" );
		this.key = key.clone();

		int Cm = 0x5a827999;
		int Mm = 0x6ed9eba1;
		int Cr = 19;
		int Mr = 17;

		int Tr[] = new int[ 24 * 8 ];
		int Tm[] = new int[ 24 * 8 ];

		// Determine the key size here, if required if keysize < 256 bytes, pad with 0 Typical key sizes => 128, 160, 192, 224, 256
		for( int i = 0; i < 24; i++ )
		{
			for( int j = 0; j < 8; j++ )
			{
				Tm[ i * 8 + j ] = Cm;
				Cm = (Cm + Mm); // mod 2^32;

				Tr[ i * 8 + j ] = Cr;
				Cr = (Cr + Mr) & 0x1f; // mod 32
			}
		}

		Binary key64 = new Binary( 64 );
		key64.set( 0, key, 0, key.size() );

		int[] wKey = new int[ 8 ];

		// now create ABCDEFGH
		for( int i = 0; i < 8; i++ )
			wKey[ i ] = key64.getIntBE( i * 4 );

		for( int i = 0; i < 12; i++ )
		{
			// KAPPA <- W2i(KAPPA)
			int i2 = i * 2 * 8;
			wKey[ 6 ] ^= F1( wKey[ 7 ], Tm[ i2 ], Tr[ i2 ] );
			wKey[ 5 ] ^= F2( wKey[ 6 ], Tm[ i2 + 1 ], Tr[ i2 + 1 ] );
			wKey[ 4 ] ^= F3( wKey[ 5 ], Tm[ i2 + 2 ], Tr[ i2 + 2 ] );
			wKey[ 3 ] ^= F1( wKey[ 4 ], Tm[ i2 + 3 ], Tr[ i2 + 3 ] );
			wKey[ 2 ] ^= F2( wKey[ 3 ], Tm[ i2 + 4 ], Tr[ i2 + 4 ] );
			wKey[ 1 ] ^= F3( wKey[ 2 ], Tm[ i2 + 5 ], Tr[ i2 + 5 ] );
			wKey[ 0 ] ^= F1( wKey[ 1 ], Tm[ i2 + 6 ], Tr[ i2 + 6 ] );
			wKey[ 7 ] ^= F2( wKey[ 0 ], Tm[ i2 + 7 ], Tr[ i2 + 7 ] );

			// KAPPA <- W2i+1(KAPPA)
			i2 = (i * 2 + 1) * 8;
			wKey[ 6 ] ^= F1( wKey[ 7 ], Tm[ i2 ], Tr[ i2 ] );
			wKey[ 5 ] ^= F2( wKey[ 6 ], Tm[ i2 + 1 ], Tr[ i2 + 1 ] );
			wKey[ 4 ] ^= F3( wKey[ 5 ], Tm[ i2 + 2 ], Tr[ i2 + 2 ] );
			wKey[ 3 ] ^= F1( wKey[ 4 ], Tm[ i2 + 3 ], Tr[ i2 + 3 ] );
			wKey[ 2 ] ^= F2( wKey[ 3 ], Tm[ i2 + 4 ], Tr[ i2 + 4 ] );
			wKey[ 1 ] ^= F3( wKey[ 2 ], Tm[ i2 + 5 ], Tr[ i2 + 5 ] );
			wKey[ 0 ] ^= F1( wKey[ 1 ], Tm[ i2 + 6 ], Tr[ i2 + 6 ] );
			wKey[ 7 ] ^= F2( wKey[ 0 ], Tm[ i2 + 7 ], Tr[ i2 + 7 ] );

			// Kr_(i) <- KAPPA
			Kr[ i * 4 ] = wKey[ 0 ] & 0x1f;
			Kr[ i * 4 + 1 ] = wKey[ 2 ] & 0x1f;
			Kr[ i * 4 + 2 ] = wKey[ 4 ] & 0x1f;
			Kr[ i * 4 + 3 ] = wKey[ 6 ] & 0x1f;

			// Km_(i) <- KAPPA
			Km[ i * 4 ] = wKey[ 7 ];
			Km[ i * 4 + 1 ] = wKey[ 5 ];
			Km[ i * 4 + 2 ] = wKey[ 3 ];
			Km[ i * 4 + 3 ] = wKey[ 1 ];
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Generate random key. Generated key will be set as current.
	 * @param keySize - 16 <= sz <= 32.
	 */
	public Binary generateKey( int keySize )
	{
		MUST( (keySize >= 16) && (keySize <= 32), "Invalid key size" );
		Binary akey = new Binary().randomSecure( keySize );
		setKey( akey );
		return akey;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary generateKey()
	{
		Binary k = new Binary().randomSecure( 16 );
		setKey( k );
		return k;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void encryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		int A = block.getIntBE( 0 );
		int B = block.getIntBE( 4 );
		int C = block.getIntBE( 8 );
		int D = block.getIntBE( 12 );

		for( int i = 0; i < 6; i++ )
		{
			int x = i * 4;
			// BETA <- Qi(BETA)
			C ^= F1( D, Km[ x ], Kr[ x ] );
			B ^= F2( C, Km[ x + 1 ], Kr[ x + 1 ] );
			A ^= F3( B, Km[ x + 2 ], Kr[ x + 2 ] );
			D ^= F1( A, Km[ x + 3 ], Kr[ x + 3 ] );
		}

		for( int i = 6; i < 12; i++ )
		{
			int x = i * 4;
			// BETA <- QBARi(BETA)
			D ^= F1( A, Km[ x + 3 ], Kr[ x + 3 ] );
			A ^= F3( B, Km[ x + 2 ], Kr[ x + 2 ] );
			B ^= F2( C, Km[ x + 1 ], Kr[ x + 1 ] );
			C ^= F1( D, Km[ x ], Kr[ x ] );
		}

		block.setIntBE( 0, A );
		block.setIntBE( 4, B );
		block.setIntBE( 8, C );
		block.setIntBE( 12, D );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void decryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		int A = block.getIntBE( 0 );
		int B = block.getIntBE( 4 );
		int C = block.getIntBE( 8 );
		int D = block.getIntBE( 12 );

		for( int i = 0; i < 6; i++ )
		{
			int x = (11 - i) * 4;
			// BETA <- Qi(BETA)
			C ^= F1( D, Km[ x ], Kr[ x ] );
			B ^= F2( C, Km[ x + 1 ], Kr[ x + 1 ] );
			A ^= F3( B, Km[ x + 2 ], Kr[ x + 2 ] );
			D ^= F1( A, Km[ x + 3 ], Kr[ x + 3 ] );
		}

		for( int i = 6; i < 12; i++ )
		{
			int x = (11 - i) * 4;
			// BETA <- QBARi(BETA)
			D ^= F1( A, Km[ x + 3 ], Kr[ x + 3 ] );
			A ^= F3( B, Km[ x + 2 ], Kr[ x + 2 ] );
			B ^= F2( C, Km[ x + 1 ], Kr[ x + 1 ] );
			C ^= F1( D, Km[ x ], Kr[ x ] );
		}

		block.setIntBE( 0, A );
		block.setIntBE( 4, B );
		block.setIntBE( 8, C );
		block.setIntBE( 12, D );
	}

}
