// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.crypt.blockcipher;

import org.denom.Binary;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;
import static org.denom.Binary.getIntLE;
import static org.denom.Binary.setIntLE;
import static java.lang.Integer.rotateLeft;

/**
 * GOST 28147-89, RFC 5830.
 */
public class GOST28147 extends BlockCipher
{
	public static final int BLOCK_SIZE = 8;
	public static final int KEY_SIZE = 32;

	// Wide sboxes -> Faster
	private final int[] wSBox0 = new int[ 256 ];
	private final int[] wSBox1 = new int[ 256 ];
	private final int[] wSBox2 = new int[ 256 ];
	private final int[] wSBox3 = new int[ 256 ];

	private int[] wKey = new int[ 8 ];

	// -----------------------------------------------------------------------------------------------------------------
	public GOST28147( byte[] sBox )
	{
		this( sBox, Bin( 32 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [32 bytes].
	 */
	public GOST28147( byte[] sBox, final Binary key  )
	{
		super.initialize( BLOCK_SIZE );
		setSBox( sBox );
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public GOST28147 setSBox( byte[] sBox )
	{
		MUST( sBox.length == 128, "SBox must be 128 bytes" );
		
		for( int i = 0; i < 256; ++i )
		{
			int iLo = i & 0x0F;
			int iHi = i >> 4;
			wSBox0[ i ] = rotateLeft(  (sBox[  16 + iHi ] << 4) | sBox[  0 + iLo ]       , 11 );
			wSBox1[ i ] = rotateLeft( ((sBox[  48 + iHi ] << 4) | sBox[ 32 + iLo ]) << 8 , 11 );
			wSBox2[ i ] = rotateLeft( ((sBox[  80 + iHi ] << 4) | sBox[ 64 + iLo ]) << 16, 11 );
			wSBox3[ i ] = rotateLeft( ((sBox[ 112 + iHi ] << 4) | sBox[ 96 + iLo ]) << 24, 11 );
		}

		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public GOST28147 clone()
	{
		GOST28147 nw = new GOST28147( new byte[128], this.key );
		System.arraycopy( this.wSBox0, 0, nw.wSBox0, 0, wSBox0.length );
		System.arraycopy( this.wSBox1, 0, nw.wSBox1, 0, wSBox1.length );
		System.arraycopy( this.wSBox2, 0, nw.wSBox2, 0, wSBox2.length );
		System.arraycopy( this.wSBox3, 0, nw.wSBox3, 0, wSBox3.length );
		return nw;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "GOST28147-89";
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary generateKey()
	{
		Binary k = new Binary().randomSecure( KEY_SIZE );
		setKey( k );
		return k;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [32 bytes].
	 */
	@Override
	public void setKey( Binary key )
	{
		MUST( key.size() == KEY_SIZE, "Invalid key size" );
		this.key = key.clone();

		for( int i = 0; i < 8; i++ )
			wKey[ i ] = key.getIntLE( i << 2 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	private int gostStep( int cm )
	{
		return wSBox3[ (cm >> 24) & 0xFF ]  |  wSBox2[ (cm >> 16) & 0xFF ]
			 | wSBox1[ (cm >>  8) & 0xFF ]  |  wSBox0[ cm & 0xFF ];
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void encryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		byte[] arr = block.getDataRef();
		int N1 = getIntLE( arr, 0 );
		int N2 = getIntLE( arr, 4 );

		for( int i = 0; i < 24; ++i )
		{
			int tmp = N1;
			N1 = N2 ^ gostStep( N1 + wKey[ i & 7 ] );
			N2 = tmp;
		}

		for( int i = 7; i > 0; i-- )
		{
			int tmp = N1;
			N1 = N2 ^ gostStep( N1 + wKey[ i ] );
			N2 = tmp;
		}

		N2 ^= gostStep( N1 + wKey[ 0 ] ); // 32 step (N1=N1)

		setIntLE( arr, 0, N1 );
		setIntLE( arr, 4, N2 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void decryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		byte[] arr = block.getDataRef();
		int N1 = getIntLE( arr, 0 );
		int N2 = getIntLE( arr, 4 );

		for( int i = 0; i < 8; ++i )
		{
			int tmp = N1;
			N1 = N2 ^ gostStep( N1 + wKey[ i ] );
			N2 = tmp;
		}

		for( int i = 31; i > 8; --i )
		{
			int tmp = N1;
			N1 = N2 ^ gostStep( N1 + wKey[ i & 7 ] ); // CM2
			N2 = tmp;
		}

		N2 ^= gostStep( N1 + wKey[ 0 ] );

		setIntLE( arr, 0, N1 );
		setIntLE( arr, 4, N2 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public Binary calcMacGost( final Binary data )
	{
		MUST( data.size() > 0, "Empty data for MAC" );

		byte[] inArr = data.getDataRef();
		byte[] mac = new byte[ BLOCK_SIZE ];

		int restLen = data.size();
		int offset = 0;
		while( restLen > 0 )
		{
			// XOR in data with prev mac
			int partSize = Math.min( BLOCK_SIZE, restLen );
			for( int i = 0; i < partSize; ++i )
				mac[ i ] ^= inArr[ offset++ ];
			restLen -= partSize;

			// Encrypt
			int N1 = getIntLE( mac, 0 );
			int N2 = getIntLE( mac, 4 );

			for( int i = 0; i < 16; i++ )
			{
				int tmp = N1;
				N1 = N2 ^ gostStep( N1 + wKey[ i & 7 ] );
				N2 = tmp;
			}

			setIntLE( mac, 0, N1 );
			setIntLE( mac, 4, N2 );
		}

		return new Binary( mac, 0, 4 );
	}
}
