// Denom.org
// bouncycastle.org

package org.denom.crypt.blockcipher;

import org.denom.Binary;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

import static java.lang.Integer.rotateLeft;
import static java.lang.Integer.rotateRight;

public class RC6 extends BlockCipher
{
	public static final int BLOCK_SIZE = 16;

	private static final int roundsNum = 20;

	private int[] S = new int[ 2 + 2 * roundsNum + 2 ];

	/**
	 * "magic constants" for wordSize 32 Pw = Odd((e-2) * 2^wordsize) Qw = Odd((o-2) * 2^wordsize)
	 * where e is the base of natural logarithms (2.718281828...) and o is the golden ratio (1.61803398...)
	 */
	private static final int P32 = 0xb7e15163;
	private static final int Q32 = 0x9e3779b9;

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public RC6()
	{
		this( Bin( 16 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 or 24 or 32 bytes]
	 */
	public RC6( final Binary key )
	{
		super.initialize( BLOCK_SIZE );
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public RC6 clone()
	{
		return new RC6( this.key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "RC6";
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 or 24 or 32 bytes].
	 */
	@Override
	public void setKey( final Binary key )
	{
		MUST( (key.size() == 16) || (key.size() == 24) || (key.size() == 32), "Invalid key size" );
		this.key = key.clone();

		// There are 3 phases to the key expansion.
		// Phase 1:
		//   Copy the secret key K[0...b-1] into an array L[0..c-1] of
		//   c = ceil(b/u), where u = wordSize/8 in little-endian order.
		//   In other words, we fill up L using u consecutive key bytes
		//   of K. Any unfilled byte positions in L are zeroed. In the
		//   case that b = c = 0, set c = 1 and L[0] = 0.
		int c = (key.size() + 3) / 4;
		if( c == 0 )
		{
			c = 1;
		}
		int[] L = new int[ (key.size() + 3) / 4 ];

		// load all key bytes into array of key dwords
		for( int i = key.size() - 1; i >= 0; i-- )
		{
			L[ i / 4 ] = (L[ i / 4 ] << 8) + key.get( i );
		}

		// Phase 2:
		//   Key schedule is placed in a array of 2+2*ROUNDS+2 = 44 dwords.
		//   Initialize S to a particular fixed pseudo-random bit pattern
		//   using an arithmetic progression modulo 2^wordsize determined
		//   by the magic numbers, Pw & Qw.
		S[ 0 ] = P32;
		for( int i = 1; i < S.length; i++ )
		{
			S[ i ] = (S[ i - 1 ] + Q32);
		}

		// Phase 3:
		//   Mix in the user's secret key in 3 passes over the arrays S & L.
		//   The max of the arrays sizes is used as the loop control
		int iter;

		if( L.length > S.length )
		{
			iter = 3 * L.length;
		}
		else
		{
			iter = 3 * S.length;
		}

		int A = 0;
		int B = 0;
		int i = 0, j = 0;

		for( int k = 0; k < iter; k++ )
		{
			A = S[ i ] = rotateLeft( S[ i ] + A + B, 3 );
			B = L[ j ] = rotateLeft( L[ j ] + A + B, A + B );
			i = (i + 1) % S.length;
			j = (j + 1) % L.length;
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Generate random key. Generated key will be set as current.
	 * @param keySize - 16 or 24 or 32 bytes.
	 */
	public Binary generateKey( int keySize )
	{
		MUST( (keySize == 16) || (keySize == 24) || (keySize == 32), "Invalid key size" );
		Binary akey = new Binary().randomSecure( keySize );
		setKey( akey );
		return akey;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary generateKey()
	{
		return generateKey( 16 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void encryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		int A = block.getIntLE( 0 );
		int B = block.getIntLE( 4 );
		int C = block.getIntLE( 8 );
		int D = block.getIntLE( 12 );

		// Do pseudo-round #0: pre-whitening of B and D
		B += S[ 0 ];
		D += S[ 1 ];

		// perform round #1,#2 ... #ROUNDS of encryption 
		for( int i = 1; i <= roundsNum; i++ )
		{
			int t = 0, u = 0;

			t = B * (2 * B + 1);
			t = rotateLeft( t, 5 );

			u = D * (2 * D + 1);
			u = rotateLeft( u, 5 );

			A ^= t;
			A = rotateLeft( A, u );
			A += S[ 2 * i ];

			C ^= u;
			C = rotateLeft( C, t );
			C += S[ 2 * i + 1 ];

			int temp = A;
			A = B;
			B = C;
			C = D;
			D = temp;
		}
		// do pseudo-round #(ROUNDS+1) : post-whitening of A and C
		A += S[ 2 * roundsNum + 2 ];
		C += S[ 2 * roundsNum + 3 ];

		block.setIntLE( 0, A );
		block.setIntLE( 4, B );
		block.setIntLE( 8, C );
		block.setIntLE( 12, D );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void decryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		int A = block.getIntLE( 0 );
		int B = block.getIntLE( 4 );
		int C = block.getIntLE( 8 );
		int D = block.getIntLE( 12 );

		// Undo pseudo-round #(ROUNDS+1) : post whitening of A and C 
		C -= S[ 2 * roundsNum + 3 ];
		A -= S[ 2 * roundsNum + 2 ];

		// Undo round #ROUNDS, .., #2,#1 of encryption 
		for( int i = roundsNum; i >= 1; i-- )
		{
			int t = 0, u = 0;

			int temp = D;
			D = C;
			C = B;
			B = A;
			A = temp;

			t = B * (2 * B + 1);
			t = rotateLeft( t, 5 );

			u = D * (2 * D + 1);
			u = rotateLeft( u, 5 );

			C -= S[ 2 * i + 1 ];
			C = rotateRight( C, t );
			C ^= u;

			A -= S[ 2 * i ];
			A = rotateRight( A, u );
			A ^= t;

		}
		// Undo pseudo-round #0: pre-whitening of B and D
		D -= S[ 1 ];
		B -= S[ 0 ];

		block.setIntLE( 0, A );
		block.setIntLE( 4, B );
		block.setIntLE( 8, C );
		block.setIntLE( 12, D );
	}
}
