// Denom.org
// bouncycastle.org

package org.denom.crypt.blockcipher;

import org.denom.Binary;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

import static java.lang.Integer.rotateLeft;

/**
 * Serpent was designed by Ross Anderson, Eli Biham and Lars Knudsen as a candidate algorithm for the NIST AES Quest.
 * http://www.cl.cam.ac.uk/~rja14/serpent.html
 * keySize = 16 or 24 or 32 bytes.
 */
public final class Serpent extends SerpentBase
{
	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public Serpent()
	{
		this( Bin(16) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 or 24 or 32 bytes].
	 */
	public Serpent( Binary key )
	{
		super.initialize( BLOCK_SIZE );
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Serpent clone()
	{
		return new Serpent( this.key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "Serpent";
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 or 24 or 32 bytes].
	 */
	@Override
	public void setKey( final Binary key )
	{
		MUST( (key.size() == 16) || (key.size() == 24) || (key.size() == 32), "Invalid key size" );
		this.key = key.clone();

		// pad key to 256 bits
		int[] kPad = new int[ 16 ];
		int off = 0;
		int length = 0;

		for( off = 0; (off + 4) < key.size(); off += 4 )
		{
			kPad[ length++ ] = key.getIntLE( off );
		}

		kPad[ length++ ] = key.getIntLE( off );
		if( length < 8 )
			kPad[ length ] = 1;

		// compute w0 to w7 from w-8 to w-1
		for( int i = 8; i < 16; i++ )
		{
			kPad[ i ] = rotateLeft( kPad[ i - 8 ] ^ kPad[ i - 5 ] ^ kPad[ i - 3 ] ^ kPad[ i - 1 ] ^ PHI ^ (i - 8), 11 );
		}

		System.arraycopy( kPad, 8, w, 0, 8 );

		setKeyCommon();
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Generate random key. Generated key will be set as current.
	 * @param keySize - 16 or 24 or 32 bytes.
	 */
	public Binary generateKey( int keySize )
	{
		MUST( (keySize == 16) || (keySize == 24) || (keySize == 32), "Invalid key size" );
		Binary akey = new Binary().randomSecure( keySize );
		setKey( akey );
		return akey;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary generateKey()
	{
		return generateKey( 16 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void encryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		X0 = block.getIntLE( 0 );
		X1 = block.getIntLE( 4 );
		X2 = block.getIntLE( 8 );
		X3 = block.getIntLE( 12 );

		encryptBlockCommon();

		block.setIntLE( 0,  w[128] ^ X0 );
		block.setIntLE( 4,  w[129] ^ X1 );
		block.setIntLE( 8,  w[130] ^ X2 );
		block.setIntLE( 12, w[131] ^ X3 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public void decryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		X0 = w[128] ^ block.getIntLE( 0 );
		X1 = w[129] ^ block.getIntLE( 4 );
		X2 = w[130] ^ block.getIntLE( 8 );
		X3 = w[131] ^ block.getIntLE( 12 );

		decryptBlockCommon();

		block.setIntLE(  0, X0 ^ w[0] );
		block.setIntLE(  4, X1 ^ w[1] );
		block.setIntLE(  8, X2 ^ w[2] );
		block.setIntLE( 12, X3 ^ w[3] );
	}
}
