// Denom.org
// bouncycastle.org

package org.denom.crypt.blockcipher;

import java.util.Arrays;

import org.denom.Binary;

import static java.lang.Integer.rotateRight;
import static org.denom.Binary.getIntBE;
import static org.denom.Binary.setIntBE;
import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

/**
 * Block cipher Shacal2, designed by Helena Handschuh and David Naccache, based on hash function
 * SHA-256, using SHA-256-Initialization-Values as data and SHA-256-Data as key.
 * A description of Shacal can be found at:
 * http://citeseerx.ist.psu.edu/viewdoc/summary?doi=10.1.1.3.4066
 * Best known cryptanalytic (Wikipedia 11.2013): Related-key rectangle attack on 44-rounds (Jiqiang Lu, Jongsung Kim).
 */
public class Shacal2 extends BlockCipher
{
	public static final int BLOCK_SIZE = 32;

	private int[] wKey = new int[ 64 ];

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public Shacal2()
	{
		this( Bin( 16 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 <= keySize <= 64], keySize % 8 = 0.
	 */
	public Shacal2( final Binary key )
	{
		super.initialize( BLOCK_SIZE );
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "Shacal2";
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Shacal2 clone()
	{
		return new Shacal2( this.key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 <= keySize <= 64], keySize % 8 = 0.
	 */
	@Override
	public void setKey( final Binary key )
	{
		MUST( (key.size() >= 16) && (key.size() <= 64) && (key.size() % 8 == 0), "Invalid key size" );
		this.key = key.clone();

		Arrays.fill( wKey, 0 );

		for( int i = 0; i < key.size() / 4; i++ )
			wKey[ i ] = key.getIntBE( i << 2 );

		for( int i = 16; i < 64; i++ )
		{
			wKey[i] = (rotateRight( wKey[i-2], 17 ) ^ rotateRight( wKey[i-2], 19 ) ^ (wKey[i-2] >>> 10))
				+ wKey[i-7] + (rotateRight( wKey[i-15], 7 ) ^ rotateRight( wKey[i-15], 18 ) ^ (wKey[i-15] >>> 3)) + wKey[i-16];
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Generate random key. Generated key will be set as current.
	 * @param keySize - [16 <= sz <= 64], sz % 8 = 0.
	 */
	public Binary generateKey( int keySize )
	{
		MUST( (keySize >= 16) && (keySize <= 64) && (keySize % 8 == 0), "Invalid key size" );
		Binary akey = new Binary().randomSecure( keySize );
		setKey( akey );
		return akey;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary generateKey()
	{
		Binary k = new Binary().randomSecure( 16 );
		setKey( k );
		return k;
	}

	// -----------------------------------------------------------------------------------------------------------------
	// SHA-256-Constants
	private final static int[] K = {
			0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5, 0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
			0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3, 0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
			0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc, 0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
			0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7, 0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
			0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13, 0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
			0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3, 0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
			0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5, 0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,
			0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208, 0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2 };

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void encryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		encryptBlock( block, block.getDataRef(), 0, block.getDataRef(), 0 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void decryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		byte[] arr = block.getDataRef();
		int b0 = getIntBE( arr,  0 );
		int b1 = getIntBE( arr,  4 );
		int b2 = getIntBE( arr,  8 );
		int b3 = getIntBE( arr, 12 );
		int b4 = getIntBE( arr, 16 );
		int b5 = getIntBE( arr, 20 );
		int b6 = getIntBE( arr, 24 );
		int b7 = getIntBE( arr, 28 );

		for( int i = 63; i >= 0; i-- )
		{
			int tmp = b0 - (rotateRight( b1, 2 ) ^ rotateRight( b1, 13 ) ^ rotateRight( b1, 22 )) - ((b1 & b2) ^ (b1 & b3) ^ (b2 & b3));
			b0 = b1;
			b1 = b2;
			b2 = b3;
			b3 = b4 - tmp;
			b4 = b5;
			b5 = b6;
			b6 = b7;
			b7 = tmp - K[ i ] - wKey[ i ] - (rotateRight( b4, 6 ) ^ rotateRight( b4, 11 ) ^ rotateRight( b4, 25 )) - ((b4 & b5) ^ ((~b4) & b6));
		}

		setIntBE( arr,  0, b0 );
		setIntBE( arr,  4, b1 );
		setIntBE( arr,  8, b2 );
		setIntBE( arr, 12, b3 );
		setIntBE( arr, 16, b4 );
		setIntBE( arr, 20, b5 );
		setIntBE( arr, 24, b6 );
		setIntBE( arr, 28, b7 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void encryptBlock( Binary block, byte[] in, int inOffset, byte[] out, int outOffset )
	{
		byte[] arr = block.getDataRef();
		int b0 = getIntBE( arr,  0 );
		int b1 = getIntBE( arr,  4 );
		int b2 = getIntBE( arr,  8 );
		int b3 = getIntBE( arr, 12 );
		int b4 = getIntBE( arr, 16 );
		int b5 = getIntBE( arr, 20 );
		int b6 = getIntBE( arr, 24 );
		int b7 = getIntBE( arr, 28 );

		for( int i = 0; i < 64; i++ )
		{
			int tmp = (rotateRight( b4, 6 ) ^ rotateRight( b4, 11 ) ^ rotateRight( b4, 25 )) + ((b4 & b5) ^ ((~b4) & b6)) + b7 + K[ i ] + wKey[ i ];
			b7 = b6;
			b6 = b5;
			b5 = b4;
			b4 = b3 + tmp;
			b3 = b2;
			b2 = b1;
			b1 = b0;
			b0 = tmp + (rotateRight( b0, 2 ) ^ rotateRight( b0, 13 ) ^ rotateRight( b0, 22 )) + ((b0 & b2) ^ (b0 & b3) ^ (b2 & b3));
		}

		setIntBE( arr,  0, b0 );
		setIntBE( arr,  4, b1 );
		setIntBE( arr,  8, b2 );
		setIntBE( arr, 12, b3 );
		setIntBE( arr, 16, b4 );
		setIntBE( arr, 20, b5 );
		setIntBE( arr, 24, b6 );
		setIntBE( arr, 28, b7 );
	}
}
