// Denom.org
// bouncycastle.org

package org.denom.crypt.blockcipher;

import org.denom.Binary;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

public class XTEA extends BlockCipher
{
	public static final int BLOCK_SIZE = 8;
	public static final int KEY_SIZE = 16;

	private static final int ROUNDS = 32;
	private static final int DELTA = 0x9E3779B9;

	private int[] sKey = new int[ 4 ];
	private int[] sum0 = new int[ 32 ];
	private int[] sum1 = new int[ 32 ];

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public XTEA()
	{
		this( Bin( KEY_SIZE ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 bytes]
	 */
	public XTEA( final Binary key )
	{
		super.initialize( BLOCK_SIZE );
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public XTEA clone()
	{
		return new XTEA( this.key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "XTEA";
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary generateKey()
	{
		Binary k = new Binary().randomSecure( KEY_SIZE );
		setKey( k );
		return k;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void setKey( Binary key )
	{
		MUST( key.size() == KEY_SIZE, "Invalid key size" );
		this.key = key.clone();

		for( int i = 0, j = 0; i < 4; ++i, j += 4 )
			sKey[ i ] = key.getIntBE( j );

		for( int i = 0, j = 0; i < ROUNDS; ++i )
		{
			sum0[ i ] = j + sKey[ j & 3 ];
			j += DELTA;
			sum1[ i ] = j + sKey[ j >>> 11 & 3 ];
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void encryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		int v0 = block.getIntBE( 0 );
		int v1 = block.getIntBE( 4 );

		for( int i = 0; i < ROUNDS; i++ )
		{
			v0 += ((v1 << 4 ^ v1 >>> 5) + v1) ^ sum0[ i ];
			v1 += ((v0 << 4 ^ v0 >>> 5) + v0) ^ sum1[ i ];
		}

		block.setIntBE( 0, v0 );
		block.setIntBE( 4, v1 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void decryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		int v0 = block.getIntBE( 0 );
		int v1 = block.getIntBE( 4 );

		for( int i = ROUNDS - 1; i >= 0; i-- )
		{
			v1 -= ((v0 << 4 ^ v0 >>> 5) + v0) ^ sum1[ i ];
			v0 -= ((v1 << 4 ^ v1 >>> 5) + v1) ^ sum0[ i ];
		}

		block.setIntBE( 0, v0 );
		block.setIntBE( 4, v1 );
	}
}
