// Denom.org
// bouncycastle.org

package org.denom.crypt.streamcipher;

import java.util.Arrays;
import org.denom.Binary;

import static java.lang.Integer.rotateRight;
import static org.denom.Binary.Bin;
import static org.denom.Binary.setIntLE;
import static org.denom.Ex.MUST;

/**
 * HC-256 - stream cipher created by Hongjun Wu.
 * The algorithm is patent-free.
 * keySize - 32 bytes.
 * iv - 32 bytes.
 */
public class HC256 extends StreamCipher
{
	int[] w = new int[ 2560 ];

	private int[] p = new int[ 1024 ];
	private int[] q = new int[ 1024 ];
	private int cnt = 0;

	private byte[] buf = new byte[ 4 ];
	private int idx = 0;

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public HC256()
	{
		this( Bin( 32 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [32 bytes]
	 */
	public HC256( final Binary key )
	{
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "HC-256";
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key - 32 bytes.
	 */
	@Override
	public void setKey( final Binary key )
	{
		MUST( key.size() == 32, "Wrong key size" );
		this.key = key.clone();
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param IV - 32 bytes.
	 */
	@Override
	public HC256 startEncrypt( final Binary iv )
	{
		init( iv );
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param IV - 32 bytes.
	 */
	@Override
	public HC256 startDecrypt( final Binary iv )
	{
		init( iv );
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void init( final Binary iv )
	{
		MUST( (iv != null) && iv.size() == 32, "Wrong IV size" );

		idx = 0;
		cnt = 0;

		Arrays.fill( w, 0 );

		for( int i = 0; i < 32; i++ )
		{
			w[ i >> 2 ] |= key.get( i ) << (8 * (i & 0x3));
		}

		for( int i = 0; i < 32; i++ )
		{
			w[ (i >> 2) + 8 ] |= iv.get( i ) << (8 * (i & 0x3));
		}

		for( int i = 16; i < 2560; i++ )
		{
			int x = w[ i - 2 ];
			int y = w[ i - 15 ];
			w[ i ] = (rotateRight( x, 17 ) ^ rotateRight( x, 19 ) ^ (x >>> 10)) + w[ i - 7 ] + (rotateRight( y, 7 ) ^ rotateRight( y, 18 ) ^ (y >>> 3)) + w[ i - 16 ] + i;
		}

		System.arraycopy( w, 512, p, 0, 1024 );
		System.arraycopy( w, 1536, q, 0, 1024 );

		for( int i = 0; i < 4096; i++ )
			step();

		cnt = 0;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public byte process( byte in )
	{
		return (byte)(in ^ getByte());
	}

	// -----------------------------------------------------------------------------------------------------------------
	private byte getByte()
	{
		if( idx == 0 )
			setIntLE( buf, 0, step() );

		byte ret = buf[ idx ];
		idx = (idx + 1) & 0x3;
		return ret;
	}

	// -----------------------------------------------------------------------------------------------------------------
	private int step()
	{
		int j = cnt & 0x3FF;
		int ret;
		if( cnt < 1024 )
		{
			int x = p[ (j - 3 & 0x3FF) ];
			int y = p[ (j - 1023 & 0x3FF) ];
			p[ j ] += p[ (j - 10 & 0x3FF) ] + (rotateRight( x, 10 ) ^ rotateRight( y, 23 )) + q[ ((x ^ y) & 0x3FF) ];

			x = p[ (j - 12 & 0x3FF) ];
			ret = (q[ x & 0xFF ] + q[ ((x >> 8) & 0xFF) + 256 ] + q[ ((x >> 16) & 0xFF) + 512 ] + q[ ((x >> 24) & 0xFF) + 768 ]) ^ p[ j ];
		}
		else
		{
			int x = q[ (j - 3 & 0x3FF) ];
			int y = q[ (j - 1023 & 0x3FF) ];
			q[ j ] += q[ (j - 10 & 0x3FF) ] + (rotateRight( x, 10 ) ^ rotateRight( y, 23 )) + p[ ((x ^ y) & 0x3FF) ];

			x = q[ (j - 12 & 0x3FF) ];
			ret = (p[ x & 0xFF ] + p[ ((x >> 8) & 0xFF) + 256 ] + p[ ((x >> 16) & 0xFF) + 512 ] + p[ ((x >> 24) & 0xFF) + 768 ]) ^ q[ j ];
		}
		cnt = cnt + 1 & 0x7FF;

		return ret;
	}
}
