// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard.cap;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import org.denom.Ex;
import org.denom.Binary;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

/**
 * CAP-file parser.
 * Считанные компоненты можно модифицировать и сохранить в новый CAP-файл.
 */
public class CapFile
{
	public static final String MANIFEST_NAME = "META-INF/MANIFEST.MF";

	// -----------------------------------------------------------------------------------------------------------------
	public Binary manifest = Bin();

	private static final int ARR_SIZE = 13;
	/**
	 * CAP-file components in binary and parsed forms.
	 * Index - Component's tag, see CapComponent.TAG_*.
	 * Elements can be null.
	 */
	public CompInfo[] comps = new CompInfo[ ARR_SIZE ];

	// -----------------------------------------------------------------------------------------------------------------
	public static class CompInfo
	{
		public String name;
		public Binary data;
		public CapComponent parsed;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * AID пакета.
	 */
	public Binary packageAID;

	/**
	 * Список с AID-ами классов, входящих в состав пакета.
	 */
	public ArrayList<Binary> classAIDs = new ArrayList<Binary>( 1 );


	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Считать из cap-файла (zip) все его компоненты.
	 * Парсинг компонентов не производится.
	 * Считанные компоненты в бинарном виде -> comps[ tag ].data.
	 * @param capPath путь к файлу.
	 * @return this;
	 */
	public CapFile load( String capPath )
	{
		loadAndParse( capPath, false );
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Распарсить cap-файл (zip) - считать и распарсить все его компоненты.
	 * Считанные компоненты в бинарном виде -> comps[ tag ].data.
	 * Каждый компонент, присутствующий в cap-файле, парсится соответствующим классом.
	 * Заполняются поля comps[ tag ].parsed. Если компонент отсутствует, то элемент = null;
	 * Компоненты в comps[ tag ].parsed можно модифицировать, а затем сохранить модифицированный CAP, см. метод 'save'.
	 * @param capPath путь к файлу.
	 * @return this;
	 */
	public CapFile parse( String capPath )
	{
		loadAndParse( capPath, true );
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	public CapComponent getComp( int tag )
	{
		CompInfo comp = comps[ tag ];
		MUST( comp != null, "Component absent" );
		return comp.parsed;
	}

	// -----------------------------------------------------------------------------------------------------------------
	private Binary readZipEntry( ZipFile zip, ZipEntry entry, byte[] tmpBuf )
	{
		Binary b = Bin();
		long entrySize = entry.getSize();
		MUST( entrySize <= Integer.MAX_VALUE, "Too large components" );
		if( entrySize > 0 )
			b.reserve( (int)entrySize );

		try( InputStream stream = zip.getInputStream( entry ) )
		{
			int len;
			while( (len = stream.read( tmpBuf )) > 0 )
				b.add( tmpBuf, 0, len );
		}
		catch( IOException ex )
		{
			throw new Ex( ex );
		}

		return b;
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void loadAndParse( String capPath, boolean parseIt )
	{
		manifest = Bin();
		comps = new CompInfo[ ARR_SIZE ];

		try( ZipFile zip = new ZipFile( capPath ) )
		{
			final Enumeration<? extends ZipEntry> entries = zip.entries();

			byte[] tmpBuf = new byte[ 128 ];

			while( entries.hasMoreElements() )
			{
				ZipEntry entry = entries.nextElement();
				String entryName = entry.getName();

				if( entryName.equalsIgnoreCase( MANIFEST_NAME ) )
					manifest = readZipEntry( zip, entry, tmpBuf );

				int tag = CapComponent.findTag( entryName );
				if( tag != -1 )
				{
					comps[ tag ] = new CompInfo();
					comps[ tag ].data = readZipEntry( zip, entry, tmpBuf );
					comps[ tag ].name = entryName;
				}
			}

			if( parseIt )
			{
				// Сначала нужно распарсить компонент 'Descriptor',
				// т.к. он используется при парсинге компонента 'Method'
				CompInfo comp = comps[ CapComponent.TAG_DESCRIPTOR ];
				MUST( comp != null, "Descriptor component is absent" );
				comp.parsed = CapComponent.parse( this, comp.name, comp.data );

				for( int i = 0; i < CapComponent.LOAD_ORDER.length; ++i )
				{
					int tag = CapComponent.LOAD_ORDER[ i ];
					comp = comps[ tag ];
					if( (comp != null) && (comp.parsed == null) )
						comp.parsed = CapComponent.parse( this, comp.name, comp.data );
				}
			}

			initAIDs();
		}
		catch( IOException exp )
		{
			throw new Ex( "CAP-file Error: \'" + capPath + "\'" );
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void initAIDs()
	{
		classAIDs.clear();

		// packageAID
		CompInfo comp = comps[ CapComponent.TAG_HEADER ];
		MUST( comp != null, "'Header.cap' is absent" );
		Binary header = comp.data;
		MUST( header.slice( 1, 2 ).asU16() == (header.size() - 3), "Wrong 'Header.cap' size" );
		int aidLen = header.get( 12 );
		packageAID = header.slice( 13, aidLen );

		// classAIDs
		comp = comps[ CapComponent.TAG_APPLET ];
		if( (comp != null) && !comp.data.empty() )
		{
			Binary applet = comp.data;
			MUST( applet.slice( 1, 2 ).asU16() == (applet.size() - 3), "Wrong 'Applet.cap' size" );
			int classCount = applet.get( 3 );

			for( int beg = 4; beg <= (applet.size() - 1) && classCount != 0; )
			{
				aidLen = applet.get( beg );
				classAIDs.add( applet.slice( beg + 1, aidLen ) );
				beg += 3 + aidLen;
			}
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Конкатенирует компоненты в порядке, рекомендованном к загрузке, см. 'CapComponent.LOAD_ORDER'.
	 * @param includeDescriptorCap - включать ли компонент Descriptor.cap
	 */
	public Binary toBinary( boolean includeDescriptorCap )
	{
		Binary bin = Bin();

		int compCount = CapComponent.LOAD_ORDER.length;
		if( !includeDescriptorCap )
			--compCount;

		for( int i = 0; i < compCount; ++i )
		{
			CompInfo comp = comps[ CapComponent.LOAD_ORDER[i] ];
			if( comp != null )
				bin.add( comp.data );
		}

		return bin;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Сохранить распарсенные компоненты в файл - создать zip-архив.
	 * ВНИМАНИЕ !!! Используются поля comps[ tag ].parsed, а не comps[ tag ].data.
	 * см. метод 'parse'.
	 * @param capPath путь к файлу.
	 */
	public void save( String capPath )
	{
		try( ZipOutputStream zip = new ZipOutputStream( new FileOutputStream( capPath ) ) )
		{
			if( manifest != null && !manifest.empty() )
			{
				zip.putNextEntry( new ZipEntry( MANIFEST_NAME ) );
				zip.write( manifest.getBytes() );
			}

			for( int i = 1; i < comps.length; ++i )
			{
				if( comps[ i ] != null )
				{
					zip.putNextEntry( new ZipEntry( comps[ i ].name ) );
					zip.write( comps[ i ].parsed.toBin().getBytes() );
				}
			}
		}
		catch( IOException ex )
		{
			throw new Ex( ex );
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	public ComponentDescriptor.MethodDescriptorInfo findMethodDescriptor( ExpFile expFile, String className, String methodName, String methodDescriptor )
	{
		ExpFile.ClassInfo classInfo = expFile.findClass( className );
		if( classInfo == null )
			return null;

		ExpFile.MethodInfo methodInfo = classInfo.findMethod( methodName, methodDescriptor );
		if( methodInfo == null )
			return null;

		ComponentDescriptor compDesc = (ComponentDescriptor)getComp( CapComponent.TAG_DESCRIPTOR );

		for( ComponentDescriptor.ClassDescriptorInfo c : compDesc.classes )
			if( c.token == classInfo.token )
				for( ComponentDescriptor.MethodDescriptorInfo m : c.methods )
					if( m.token == methodInfo.token )
						return m;

		return null;
	}

	// -----------------------------------------------------------------------------------------------------------------
	public ComponentMethod.MethodInfo findMethodInfo( ExpFile expFile, String className, String methodName, String methodDescriptor )
	{
		ComponentDescriptor.MethodDescriptorInfo md = findMethodDescriptor( expFile, className, methodName, methodDescriptor );
		if( md == null )
			return null;

		ComponentMethod compMethod = (ComponentMethod)getComp( CapComponent.TAG_METHOD );

		for( ComponentMethod.MethodInfo m : compMethod.methods )
			if( m.methodDescr == md )
				return m;

		return null;
	}
}
