// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard.cap;

import java.util.ArrayList;
import org.denom.Binary;
import org.denom.smartcard.cap.ComponentConstantPool.ClassRef;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

/**
 * CAP-file component - Class.
 */
public class ComponentClass extends CapComponent
{
	public int tag;
	public int size;

	public ArrayList< TypeDescriptor > signaturePool;
	public ArrayList< InterfaceInfo > interfaces;
	public ArrayList< ClassInfo > classes;

	// -----------------------------------------------------------------------------------------------------------------
	public ComponentClass( CapFile cap, String fullName, final Binary b )
	{
		super( cap, TAG_CLASS, fullName );

		int offs = 0;

		tag = b.get( offs++ );
		size = b.getU16( offs ); offs += 2;

		int minorVersion = ((ComponentHeader)cap.getComp( TAG_HEADER )).minorVersion;

		signaturePool = new ArrayList<>();
		interfaces = new ArrayList<>();
		classes = new ArrayList<>();

		if( minorVersion == 2 )
		{
			// in bytes
			int sigPoolLen = b.getU16( offs ); offs += 2;
		
			int startOffset = offs;
			for( int len = 0; len < sigPoolLen; len = offs - startOffset )
			{
				TypeDescriptor td = new TypeDescriptor();
				offs = td.parse( b, offs );
				signaturePool.add( td );
			}
		}

		while( offs < b.size() )
		{
			int flags = b.get( offs );
			if( (flags & ACC_INTERFACE) != 0 )
			{
				InterfaceInfo intf = new InterfaceInfo();
				offs = intf.parse( b, offs );
				interfaces.add( intf );
			}
			else
			{
				ClassInfo cl = new ClassInfo();
				offs = cl.parse( b, offs );
				classes.add( cl );
			}
		}

		MUST( (tag == TAG_CLASS) && (size == (b.size() - 3))
				&& (offs == b.size()), "Wrong component Class" );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public Binary toBin()
	{
		Binary b = Bin().reserve( size + 3 );

		b.add( tag );
		b.addU16( size );

		int minorVersion = ((ComponentHeader)cap.getComp( TAG_HEADER )).minorVersion;
		if( minorVersion == 2 )
		{
			Binary sp = Bin().reserve( 100 );
			for( TypeDescriptor td : signaturePool )
				td.toBin( sp );
			b.addU16( sp.size() );
			b.add( sp );
		}

		for( InterfaceInfo intf : interfaces )
			intf.toBin( b );

		for( ClassInfo cl : classes )
			cl.toBin( b );

		return b;
	}

	// =================================================================================================================
	public static final byte ACC_INTERFACE = 0x8;
	public static final byte ACC_SHAREABLE = 0x4;
	public static final byte ACC_REMOTE    = 0x2;

	// =================================================================================================================
	public static class TypeDescriptor
	{
		public int nibbleCount;
		public Binary type;

		public int parse( Binary b, int offs )
		{
			nibbleCount = b.get( offs++ );
			int sz = (nibbleCount + 1) / 2;
			type = b.slice( offs, sz );
			offs += sz;
			return offs;
		}

		public void toBin( Binary b )
		{
			b.add( nibbleCount );
			b.add( type );
		}
	}

	// =================================================================================================================
	public static class InterfaceInfo
	{
		public int flags;
		public ClassRef[] superInterfaces;
		Binary interfaceName;

		public int parse( Binary b, int offs )
		{
			int b0 = b.get( offs++ );

			flags = b0 >> 4;
			int interfaceCount = b0 & 0x0F;

			superInterfaces = new ClassRef[ interfaceCount ];
			for( int i = 0; i < interfaceCount; i++ )
			{
				superInterfaces[ i ] = new ClassRef();
				offs = superInterfaces[ i ].parse( b, offs );
			}

			if( (flags & ACC_REMOTE) != 0 )
			{
				int len = b.get( offs++ );
				interfaceName = b.slice( offs, len );
				offs += len;
			}

			return offs;
		}

		public void toBin( Binary b )
		{
			b.add( (flags << 4) | superInterfaces.length );
			for( ClassRef cr : superInterfaces )
				cr.toBin( b );

			if( interfaceName != null )
			{
				b.add( interfaceName.size() );
				b.add( interfaceName );
			}
		}
	}

	// =================================================================================================================
	public static class ImplementedInterfaceInfo
	{
		public ClassRef intf;
		public Binary index;

		public int parse( Binary b, int offs )
		{
			intf = new ClassRef();
			offs = intf.parse( b, offs );

			int count = b.get( offs++ );
			index = b.slice( offs, count );
			offs += count;
			return offs;
		}

		public void toBin( Binary b )
		{
			intf.toBin( b );
			b.add( index.size() );
			b.add( index );
		}
	}

	// =================================================================================================================
	public static class RemoteMethodInfo
	{
		public int remoteMethodHash;
		public int signatureOffset;
		public int virtualMethodToken;

		public int parse( Binary b, int offs )
		{
			remoteMethodHash = b.getU16( offs ); offs += 2;
			signatureOffset = b.getU16( offs ); offs += 2;
			virtualMethodToken = b.get( offs++ );
			return offs;
		}

		public void toBin( Binary b )
		{
			b.addU16( remoteMethodHash );
			b.addU16( signatureOffset );
			b.add( virtualMethodToken );
		}
	}

	// =================================================================================================================
	public static class RemoteInterfaceInfo
	{
		public RemoteMethodInfo[] remoteMethods;
		public Binary hashModifier;
		public Binary className;
		public ClassRef[] remoteInterfaces;

		public int parse( Binary b, int offs )
		{
			int count = b.get( offs++ );
			remoteMethods = new RemoteMethodInfo[ count ];
			for( int i = 0; i < count; ++i )
			{
				remoteMethods[ i ] = new RemoteMethodInfo();
				offs = remoteMethods[ i ].parse( b, offs );
			}

			int len = b.get( offs++ );
			hashModifier = b.slice( offs, len );
			offs += len;

			len = b.get( offs++ );
			className = b.slice( offs, len );
			offs += len;

			count = b.get( offs++ );
			remoteInterfaces = new ClassRef[ count ];
			for( int i = 0; i < count; ++i )
			{
				remoteInterfaces[ i ] = new ClassRef();
				offs = remoteInterfaces[ i ].parse( b, offs );
			}

			return offs;
		}

		public void toBin( Binary b )
		{
			b.add( remoteMethods.length );
			for( RemoteMethodInfo m : remoteMethods )
				m.toBin( b );

			b.add( hashModifier.size() );
			b.add( hashModifier );

			b.add( className.size() );
			b.add( className );

			b.add( remoteInterfaces.length );
			for( ClassRef cl : remoteInterfaces )
				cl.toBin( b );
		}
	}

	// =================================================================================================================
	public static class ClassInfo
	{
		public int flags;
		public ClassRef superClass;

		public int declaredInstanceSize;
		public int firstReferenceToken;
		public int referenceCount;
		public int publicMethodTableBase;
		public int publicMethodTableCount;
		public int packageMethodTableBase;
		public int packageMethodTableCount;
		public int[] publicVMT;
		public int[] packageVMT;

		public ImplementedInterfaceInfo[] interfaces;

		public RemoteInterfaceInfo remoteInterfaces;


		public int parse( Binary b, int offs )
		{
			int b0 = b.get( offs++ );

			flags = b0 >> 4;
			int interfaceCount = b0 & 0x0F;

			superClass = new ClassRef();
			offs = superClass.parse( b, offs );

			declaredInstanceSize = b.get( offs++ );
			firstReferenceToken = b.get( offs++ );
			referenceCount = b.get( offs++ );
			publicMethodTableBase = b.get( offs++ );
			publicMethodTableCount = b.get( offs++ );
			packageMethodTableBase = b.get( offs++ );
			packageMethodTableCount = b.get( offs++ );

			publicVMT = new int[ publicMethodTableCount ];
			for( int i = 0; i < publicVMT.length; ++i )
			{
				publicVMT[ i ] = b.getU16( offs ); offs += 2;
			}

			packageVMT = new int[ packageMethodTableCount ];
			for( int i = 0; i < packageVMT.length; ++i )
			{
				packageVMT[ i ] = b.getU16( offs ); offs += 2;
			}

			interfaces = new ImplementedInterfaceInfo[ interfaceCount ];
			for( int i = 0; i < interfaces.length; ++i )
			{
				interfaces[ i ] = new ImplementedInterfaceInfo();
				offs = interfaces[ i ].parse( b, offs );
			}

			if( (flags & ACC_REMOTE) != 0 )
			{
				remoteInterfaces = new RemoteInterfaceInfo();
				offs = remoteInterfaces.parse( b, offs );
			}

			return offs;
		}


		public void toBin( Binary b )
		{
			b.add( (flags << 4) | interfaces.length );
			superClass.toBin( b );

			b.add( declaredInstanceSize );
			b.add( firstReferenceToken );
			b.add( referenceCount );
			b.add( publicMethodTableBase );
			b.add( publicMethodTableCount );
			b.add( packageMethodTableBase );
			b.add( packageMethodTableCount );

			for( int n : publicVMT )
				b.addU16( n );

			for( int n : packageVMT )
				b.addU16( n );

			for( ImplementedInterfaceInfo intf : interfaces )
				intf.toBin( b );

			if( remoteInterfaces != null )
				remoteInterfaces.toBin( b );
		}
	}

}
