// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package kernel8;

import org.denom.*;
import org.denom.crypt.blockcipher.*;
import org.denom.crypt.ec.ECAlg;
import org.denom.crypt.ec.Fp.custom.Secp256r1;
import org.denom.crypt.hash.SHA256;
import org.denom.format.BerTLV;
import org.denom.log.*;
import org.denom.smartcard.emv.EmvCrypt;
import org.denom.smartcard.emv.TagEmv;
import org.denom.smartcard.emv.kernel8.Kernel8Crypt;

import static org.denom.Binary.*;
import static org.denom.Ex.MUST;

/**
 * Contactless Book E, Cryptography Worked Examples v1.0.1,  6.1  BDH Key Agreement
 */
public class TestEmvCryptBookE
{
	public static void main( String[] args )
	{
		ILog log = new LogConsole();
		SHA256 sha256 = new SHA256();

		// Секретный ключ карты
		Binary Dc = Bin("9CCCF19E58228AED9CEEB7D48F69D3B37EE56F416D642E13D5F46FC9D198E08F");

		// Случайное число карты - blinding factor
		Binary r = Bin("8C5AE5C30D3164A755D101C50646F405A06761562EC0CCF940D2B6E67CE2F1F8");

		// Секретный ключ терминала
		Binary Dk = Bin("7D6FD7D5B428CF88A1A8B0F2A6A0DC563B6882CBCB09DB8C63E05E78B008C2E7");

		ECAlg algIcc = new ECAlg( new Secp256r1() );
		algIcc.setPrivate( Dc );
		ECAlg algKernel = algIcc.clone();
		algKernel.setPrivate( Dk );

		// Публичный ключ карты
		Binary Qc = algIcc.getPublic( false );
		MUST( Qc.equals( "04 43CA1837F6B4321CA70262902037EFCE790DC583828AEA628FFAAEFC08618658 605F7182F9C750D70FE18DDDE9D8006F09A2F4732E538DBBCE6DF40B50B58C71" ),
				"Wrong Qc" );

		// Публичный ключ терминала
		Binary Qk = algKernel.getPublic( false );
		MUST( Qk.equals( "04 B3B037BB4E599B3E7392332CE9DEC1D388EF551C9B8FDE20D5ABAE5B8464707E 2966B53449B958462AE970FC16462BB1DDF83D1A7257BBEBA596E8D40DACFE02" ),
				"Wrong Qk" );

		Binary Pc = Kernel8Crypt.BDHCalcPc( algIcc, r );
		MUST( Pc.equals( "D4D37071C5EC5AC817FE5B7500E623A1323DEC9241DE35099834FB0618A062EC" ), "Wrong Pc" );

		Binary z = Kernel8Crypt.BDHCalcZ( algKernel, Pc );
		Binary z2 = Kernel8Crypt.BDHCalcZ( algKernel, Dc, r );
		MUST( z.equals( "154CC3AAB0FCCC8F94FF3CFEE1BFD5AE4498AFF011CA027D1860FFF10F7C296D" ), "Wring z" );
		MUST( z2.equals( z ), "Wring z2" );

		Binary Kd = Kernel8Crypt.BDHCalcKd( z );
		MUST( Kd.equals( "602A2FC322991249E47991E4B6952D38" ), "Wrong Kd" );
		Binary SKc = Kernel8Crypt.BDHCalcSKc( Kd );
		MUST( SKc.equals("88655CFD79DF9E9DDDEAF9EC0C538DC5"), "Wrong SKc" );
		Binary SKi = Kernel8Crypt.BDHCalcSKi( Kd );
		MUST( SKi.equals("642373F56192B09B132C7E024164D3A7"), "Wrong SKi" );

		AES aesSKc = new AES( SKc );
		AES aesSKi = new AES( SKi );

		Binary crypt = Kernel8Crypt.cryptCTR( aesSKc, 0x8000, r );
		MUST( crypt.equals( "4A7653A86A6AE421DB875BF695F31C8631C1EDB721EBFCBB3057C87DB03EEA7A" ), "Wrong AES-CTR" );

		crypt = Kernel8Crypt.cryptCTR( aesSKc, 0x8003, Bin("9F8113081122334455AABBCC") );
		MUST( crypt.equals( "44BD326A7CF2529D815B0868" ), "Wrong AES-CTR" );

		Binary cmac = aesSKi.calcCMAC( Bin("80049F81120411FF00EE"), null );
		MUST( cmac.equals( "480814635965B2E55A689163EF3BDCEF" ), "Wrong AES-CMAC" );

		Binary input = Bin("00000280000110FF0000515B6EEC6CAA4A0B7445902CA236965CF9C24E002F94ED6267F258173B39BAF32C7902D36621\n"
				+ "252EE62A1BF6FA0787F19EEB523D716122CFC88704B067D0AB7A00000001000000000000100008400000000082084024\n"
				+ "11150097576168000000F7808000000000000000208004514CD3B3C700440058000F9F2701809F360200029F81020100\n"
				+ "9F101A2211A080032420000000000000000000000000000000000000005866EFC183F905F0797A7F41811B32A71EC2F8\n"
				+ "428F69E775A46943133839A9D0");
		Binary cmacp = Kernel8Crypt.calcAesCmacPlus( aesSKi, input );
		MUST( cmacp.equals( "38264BCF18380F10C213C62F1D4C7DB2" ), "Wrong AES-CMAC+" );

		Binary sda = Bin( "5A08541333900000151357125413339000001513D29122010000000000009F810B2043CA1837F6B4321CA70262902037\n"
				+ "EFCE790DC583828AEA628FFAAEFC086186585F24032912319F420209789F0702FF005F3401019F810A038C9F068C1F9F\n"
				+ "02069F03069F1A0295055F2A029A039C019F37049F34039F1D089F810C029F0607A00000000410100103" );
		Binary sdaHash = sha256.calc( sda );
		MUST( sdaHash.equals( "5866EFC183F905F0797A7F41811B32A71EC2F8428F69E775A46943133839A9D0" ) );

		Binary PDOLValues = Bin( "0280000110FF0000515B6EEC6CAA4A0B7445902CA236965CF9C24E002F94ED6267F258173B39BAF32C7902D36621252E\n"
				+ "E62A1BF6FA0787F19EEB523D716122CFC88704B067D0AB7A" );
		Binary CDOL1RelatedData = Bin( "0000000100000000000010000840000000008208402411150097576168000000F7808000000000000000" );
		Binary terminalRREntropy = Bin( "20800451" );
		Binary lastERRDResponse = Bin( "4CD3B3C700440058000F" );
		Binary genAcResponse = BerTLV.Tlv( TagEmv.ResponseMessageTemplateFormat2,
				"9F2701809F360200029F810201009F101A2211A08003242000000000000000000000000000000000000000" );

		Binary data = Kernel8Crypt.formDataForIADMac( PDOLValues, CDOL1RelatedData, terminalRREntropy, lastERRDResponse, genAcResponse, sdaHash );
		Binary iadMac = Kernel8Crypt.calcIADMac( aesSKi, data );
		MUST( iadMac.equals( "38264BCF18380F10" ),"Wrong IAD-MAC" );


		// Session key AC  -- 3DES
		TripleDES MKdes = new TripleDES( Bin("4F5276A285D36175DC4F516BBF80CB1A") );
		Binary SKac = EmvCrypt.calcSKac( MKdes, Bin("0001") );
		MUST( SKac.equals( "5B10C70AFEE94975A345C69888048FF9" ), "Wrong 3DES SKac" );

		Binary acInputData = Bin( "0000000100000000000010000840000000008208402411150097576168010B00012211A08003\n"
				+ "242000000000000000000000004E25BD7ED9AA897A" );

		Binary ac = EmvCrypt.calcAC( new TripleDES( SKac ), acInputData );
		MUST( ac.equals( "D57662E3C49381E9" ), "Wrong 3DES AC" );


		// Session key AES128
		AES MKaes128 = new AES( Bin("2EF6E07ECBA86BCF3C3CFF7BBEBE6F38") );
		SKac = EmvCrypt.calcSKac( MKaes128, Bin("0001") );
		MUST( SKac.equals( "89F7B697A028A93345BE7A409665B9A4" ), "Wrong AES128 SKac" );

		ac = EmvCrypt.calcAC( new AES( SKac ), acInputData );
		MUST( ac.equals( "45B53B28AE92F9B8" ), "Wrong AES128 AC" );


		// Session key AES192
		AES MKaes192 = new AES( Bin("7481B0525D05D393D6DCBADD333C6BC514087C6BF5FA10C4") );
		SKac = EmvCrypt.calcSKac( MKaes192, Bin("0001") );
		MUST( SKac.equals( "78429FD2061D24B1F8830B2C91D3ED95ED9D4B069A7C2707" ), "Wrong AES192 SKac" );

		ac = EmvCrypt.calcAC( new AES( SKac ), acInputData );
		MUST( ac.equals( "A32E7F58F4B7C703" ), "Wrong AES192 AC" );


		// Session key AES256
		AES MKaes256 = new AES( Bin("14D63F23982740AC65B482BAF5913092D8132BAA4143A24D3CF437232711A507") );
		SKac = EmvCrypt.calcSKac( MKaes256, Bin("0001") );
		MUST( SKac.equals( "A99C5840A0CBFBF093DEA740FFFFF5A241BFE9472968CC6E0B9EC76BA280CE0F" ), "Wrong AES256 SKac" );

		ac = EmvCrypt.calcAC( new AES( SKac ), acInputData );
		MUST( ac.equals( "E9CA46A1B7533CD9" ), "Wrong AES256 AC" );


		ac = Bin("3AC508533D4F7C87");
		Binary edaMac = Kernel8Crypt.calcEDAMac( aesSKi, ac, iadMac );
		MUST( edaMac.equals( "7A56EF21900D02EE" ),"Wrong EDA-MAC" );

		//log.writeln( "EDA-MAC: " + edaMac.Hex() );

		// READ DATA
		aesSKc = new AES( Bin("88655CFD79DF9E9DDDEAF9EC0C538DC5") );
		aesSKi = new AES( Bin("642373F56192B09B132C7E024164D3A7") );
		Binary plain = Bin("9F8113081122334455AABBCC");
		crypt = Kernel8Crypt.encryptReadData( aesSKc, aesSKi, 0x8003, plain );
		MUST( crypt.equals( Bin("44BD326A7CF2529D815B0868 197B3882007367D1") ), "Wrong encryption READ DATA" );

		Binary plain2 = Kernel8Crypt.decryptReadData( aesSKc, aesSKi, 0x8003, crypt );
		MUST( plain2.equals( plain ), "Wrong decryption READ DATA" );

		// WRITE DATA
		plain = Bin("9F81120411FF00EE");
		crypt = Kernel8Crypt.cryptWriteData( aesSKc, 0x0000, plain );
		MUST( crypt.equals( Bin("AA0DC8BFDD582802") ), "Wrong encryption WRITE DATA" );

		Binary mac = Kernel8Crypt.calcDataEnvelopeMac( aesSKi, 0x8004, plain );
		MUST( mac.equals( Bin("480814635965B2E5") ), "Wrong DATA ENVELOPE MAC" );

		log.writeln( "All OK" );
	}
}
