// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package testblockcipher;

import org.denom.*;
import org.denom.crypt.blockcipher.*;

import static org.denom.Binary.Bin;

public class TestAES extends CheckBlockCipher
{
	public static void main( String[] args ) { new TestAES(); }

	Binary testData = Bin(
			  "6B C1 BE E2  2E 40 9F 96  E9 3D 7E 11  73 93 17 2A"
			+ "AE 2D 8A 57  1E 03 AC 9C  9E B7 6F AC  45 AF 8E 51"
			+ "30 C8 1C 46  A3 5C E4 11  E5 FB C1 19  1A 0A 52 EF"
			+ "F6 9F 24 45  DF 4F 9B 17  AD 2B 41 7B  E6 6C 37 10" );

	// -----------------------------------------------------------------------------------------------------------------
	public TestAES()
	{
		AES alg = new AES();

		check( alg, "80000000000000000000000000000000", "00000000000000000000000000000000", "0EDD33D3C621E546455BD8BA1418BEC8" );
		check( alg, "00000000000000000000000000000080", "00000000000000000000000000000000", "172AEAB3D507678ECAF455C12587ADB7" );

		check( alg, "000000000000000000000000000000000000000000000000", "80000000000000000000000000000000", "6CD02513E8D4DC986B4AFE087A60BD0C" );

		check( alg, "0000000000000000000000000000000000000000000000000000000000000000", "80000000000000000000000000000000", "DDC6BF790C15760D8D9AEB6F9A75FD4E" );

		// -----------------------------------------------------------------------------------------------------------------
		checkMonteCarlo( alg, 10000, "00000000000000000000000000000000", "00000000000000000000000000000000", "C34C052CC0DA8D73451AFE5F03BE297F" );
		checkMonteCarlo( alg, 10000, "5F060D3716B345C253F6749ABAC10917", "355F697E8B868B65B25A04E18D782AFA", "ACC863637868E3E068D2FD6E3508454A" );

		checkMonteCarlo( alg, 10000, "AAFE47EE82411A2BF3F6752AE8D7831138F041560631B114", "F3F6752AE8D7831138F041560631B114", "77BA00ED5412DFF27C8ED91F3C376172" );

		checkMonteCarlo( alg, 10000, "28E79E2AFC5F7745FCCABE2F6257C2EF4C4EDFB37324814ED4137C288711A386",
				"C737317FE0846F132B23C8C2A672CE22", "E58B82BFBA53C0040DC610C642121168" );

		// -----------------------------------------------------------------------------------------------------------------
		String dataHex = "AAFE47EE82411A2BF3F6752AE8D7831138F041560631B114F3F6752AE8D7831138F041560631B1145A01020304050607";
		String keyHex = "5F060D3716B345C253F6749ABAC10917";
		Binary iv = Bin("000102030405060708090a0b0c0d0e0f");

		check( alg, CryptoMode.CBC, AlignMode.NONE, keyHex, null,
				dataHex, "a444a9a4d46eb30cb7ed34d62873a89f8fdf2bf8a54e1aeadd06fd85c9cb46f021ee7cd4f418fa0bb72e9d07c70d5d20" );

		check( alg, CryptoMode.CBC, AlignMode.NONE, keyHex, iv,
				dataHex, "585681354f0e01a86b32f94ebb6a675045d923cf201263c2aaecca2b4de82da0edd74ca5efd654c688f8a58e61955b11" );

		check( alg, CryptoMode.CFB, AlignMode.NONE, keyHex, null,
				dataHex, "82a1744e8ebbd053ca72362d5e5703264b4182de3208c374b8ac4fa36af9c5e5f4f87d1e3b67963d06acf5eb13914c90" );

		check( alg, CryptoMode.CFB, AlignMode.NONE, keyHex, iv,
				dataHex, "146cbb581d9e12c3333dd9c736fbb9303c8a3eb5185e2809e9d3c28e25cc2d2b6f5c11ee28d6530f72c412b1438a816a" );

		check( alg, CryptoMode.OFB, AlignMode.NONE, keyHex, null,
				dataHex, "82a1744e8ebbd053ca72362d5e5703261ebf1fdbec05e57b3465b583132f84b43bf95b2c89040ad1677b22d42db69a7a" );

		check( alg, CryptoMode.OFB, AlignMode.NONE, keyHex, iv,
				dataHex, "146cbb581d9e12c3333dd9c736fbb9309ea4c2a7696c84959a2dada49f2f1c5905db1f0cec3a31acbc4701e74ab05e1f" );

		// -----------------------------------------------------------------------------------------------------------------
		// A Specification for Rijndael, the AES Algorithm
		check( alg, "2b7e151628aed2a6abf7158809cf4f3c", "3243f6a8885a308d313198a2e0370734", "3925841d02dc09fbdc118597196a0b32" );
		check( alg, "2b7e151628aed2a6abf7158809cf4f3c762e7160f38b4da5", "3243f6a8885a308d313198a2e0370734", "f9fb29aefc384a250340d833b87ebc00" );
		check( alg, "2b7e151628aed2a6abf7158809cf4f3c762e7160f38b4da56a784d9045190cfe", "3243f6a8885a308d313198a2e0370734", "1a6e6c2c662e7da6501ffb62bc9e93f3" );

		// -----------------------------------------------------------------------------------------------------------------
		// sp800-38a
		String key128 = "2b7e151628aed2a6abf7158809cf4f3c";
		String key192 = "8e73b0f7da0e6452c810f32b809079e562f8ead2522c6b7b";
		String key256 = "603deb1015ca71be2b73aef0857d77811f352c073b6108d72d9810a30914dff4";

		check( alg, CryptoMode.ECB, AlignMode.NONE, key128, null,
				testData.Hex(), "3ad77bb40d7a3660a89ecaf32466ef97 f5d3d58503b9699de785895a96fdbaaf 43b1cd7f598ece23881b00e3ed030688 7b0c785e27e8ad3f8223207104725dd4" );

		check( alg, CryptoMode.ECB, AlignMode.NONE, key192, null,
				testData.Hex(), "bd334f1d6e45f25ff712a214571fa5cc 974104846d0ad3ad7734ecb3ecee4eef ef7afd2270e2e60adce0ba2face6444e 9a4b41ba738d6c72fb16691603c18e0e" );

		check( alg, CryptoMode.ECB, AlignMode.NONE, key256, null,
				testData.Hex(), "f3eed1bdb5d2a03c064b5a7e3db181f8 591ccb10d410ed26dc5ba74a31362870 b6ed21b99ca6f4f9f153e7b1beafed1d 23304b7a39f9f3ff067d8d8f9e24ecc7" );

		check( alg, CryptoMode.CBC, AlignMode.NONE, key128, iv,
				testData.Hex(), "7649abac8119b246cee98e9b12e9197d 5086cb9b507219ee95db113a917678b2 73bed6b8e3c1743b7116e69e22229516 3ff1caa1681fac09120eca307586e1a7" );

		check( alg, CryptoMode.CBC, AlignMode.NONE, key192, iv,
				testData.Hex(), "4f021db243bc633d7178183a9fa071e8 b4d9ada9ad7dedf4e5e738763f69145a 571b242012fb7ae07fa9baac3df102e0 08b0e27988598881d920a9e64f5615cd" );

		check( alg, CryptoMode.CBC, AlignMode.NONE, key256, iv,
				testData.Hex(), "f58c4c04d6e5f1ba779eabfb5f7bfbd6 9cfc4e967edb808d679f777bc6702c7d 39f23369a9d9bacfa530e26304231461 b2eb05e2c39be9fcda6c19078c6a9d1b" );

		check( alg, CryptoMode.CFB, AlignMode.NONE, key128, iv,
				testData.Hex(), "3b3fd92eb72dad20333449f8e83cfb4a c8a64537a0b3a93fcde3cdad9f1ce58b 26751f67a3cbb140b1808cf187a4f4df c04b05357c5d1c0eeac4c66f9ff7f2e6" );

		// check not aligned size (25 bytes)
		check( alg, CryptoMode.CFB, AlignMode.NONE, key128, iv,
				testData.first(25).Hex(), "3b3fd92eb72dad20333449f8e83cfb4a c8a64537a0b3a93fcd" );

		check( alg, CryptoMode.CFB, AlignMode.NONE, key192, iv,
				testData.Hex(), "cdc80d6fddf18cab34c25909c99a4174 67ce7f7f81173621961a2b70171d3d7a 2e1e8a1dd59b88b1c8e60fed1efac4c9 c05f9f9ca9834fa042ae8fba584b09ff" );

		check( alg, CryptoMode.CFB, AlignMode.NONE, key256, iv,
				testData.Hex(), "dc7e84bfda79164b7ecd8486985d3860 39ffed143b28b1c832113c6331e5407b df10132415e54b92a13ed0a8267ae2f9 75a385741ab9cef82031623d55b1e471" );

		check( alg, CryptoMode.OFB, AlignMode.NONE, key128, iv,
				testData.Hex(), "3b3fd92eb72dad20333449f8e83cfb4a 7789508d16918f03f53c52dac54ed825 9740051e9c5fecf64344f7a82260edcc 304c6528f659c77866a510d9c1d6ae5e" );

		// check not aligned size (37 bytes)
		check( alg, CryptoMode.OFB, AlignMode.NONE, key128, iv,
				testData.first(37).Hex(), "3b3fd92eb72dad20333449f8e83cfb4a 7789508d16918f03f53c52dac54ed825 9740051e9c" );

		check( alg, CryptoMode.OFB, AlignMode.NONE, key192, iv,
				testData.Hex(), "cdc80d6fddf18cab34c25909c99a4174 fcc28b8d4c63837c09e81700c1100401 8d9a9aeac0f6596f559c6d4daf59a5f2 6d9f200857ca6c3e9cac524bd9acc92a" );

		check( alg, CryptoMode.OFB, AlignMode.NONE, key256, iv,
				testData.Hex(), "dc7e84bfda79164b7ecd8486985d3860 4febdc6740d20b3ac88f6ad82a4fb08d 71ab47a086e86eedf39d1c5bba97c408 0126141d67f37be8538f5a8be740e484" );

		// -----------------------------------------------------------------------------------------------------------------
		// sp800-38a
		String sv = "f0f1f2f3f4f5f6f7f8f9fafbfcfdfeff";
		checkCTR( alg, key128, sv, testData.Hex(),
				"874d6191b620e3261bef6864990db6ce 9806f66b7970fdff8617187bb9fffdff 5ae4df3edbd5d35e5b4f09020db03eab 1e031dda2fbe03d1792170a0f3009cee" );

		checkCTR( alg, key192, sv, testData.Hex(),
				"1abc932417521ca24f2b0459fe7e6e0b 090339ec0aa6faefd5ccc2c6f4ce8e94 1e36b26bd1ebc670d1bd1d665620abf7 4f78a7f6d29809585a97daec58c6b050" );

		checkCTR( alg, key256, sv, testData.Hex(),
				"601ec313775789a5b7a7f504bbf3d228 f443e3ca4d62b59aca84e990cacaf5c5 2b0930daa23de94ce87017ba2d84988d dfc9c58db67aada613c2dd08457941a6" );

		// EMV Contactless Book E, Cryptography Worked Examples v1.0.1, 6.1  BDH Key Agreement, Blinding factor validation
		String keyBookE = "88655CFD79DF9E9DDDEAF9EC0C538DC5";
		checkCTR( alg, keyBookE,
				"80000000000000000000000000000000",
				"8C5AE5C30D3164A755D101C50646F405A06761562EC0CCF940D2B6E67CE2F1F8",
				"4A7653A86A6AE421DB875BF695F31C8631C1EDB721EBFCBB3057C87DB03EEA7A" );

		checkCTR( alg, keyBookE,
				"80010000000000000000000000000000",
				"5A08541333900000151357125413339000001513D29122010000000000005F24032912319F420209789F0702FF005F3401019F810A038C9F06",
				"0179C459B24A72C6D9359091CFE515E67FC1F8CD1C980DDB5585F82D6C4234A409185CC388AAF6E9D7636FC6E6A1A702F47625A5951A7908D0" );

		checkCTR( alg, keyBookE,
				"80030000000000000000000000000000",
				"9F8113081122334455AABBCC",
				"44BD326A7CF2529D815B0868" );

		// -----------------------------------------------------------------------------------------------------------------
		checkMode( alg, CryptoMode.ECB, AlignMode.NONE );
		checkMode( alg, CryptoMode.CBC, AlignMode.NONE );
		checkMode( alg, CryptoMode.CFB, AlignMode.NONE );
		checkMode( alg, CryptoMode.OFB, AlignMode.NONE );

		checkMode( alg, CryptoMode.ECB, AlignMode.BLOCK );
		checkMode( alg, CryptoMode.CBC, AlignMode.BLOCK );
		checkMode( alg, CryptoMode.CFB, AlignMode.BLOCK );
		checkMode( alg, CryptoMode.OFB, AlignMode.BLOCK );

		checkMAC( alg, AlignMode.NONE );
		checkMAC( alg, AlignMode.BLOCK );

		// -----------------------------------------------------------------------------------------------------------------
		jceCompareEncrypt( alg, "AES", CryptoMode.ECB );
		jceCompareEncrypt( alg, "AES", CryptoMode.CBC );
		jceCompareEncrypt( alg, "AES", CryptoMode.CFB );
		jceCompareEncrypt( alg, "AES", CryptoMode.OFB );

		long t = measure( alg, CryptoMode.CBC, ITERATIONS, DATA_SIZE );
		log.writeln( "Time AES     : " + t + " ms" );

		t = jceMeasureTime( "AES", CryptoMode.CBC, ITERATIONS, Bin().random( 16 ), Bin().random( 16 ), Bin().random( DATA_SIZE ) );
		log.writeln( "Time AES JCE : " + t + " ms" );

		// -----------------------------------------------------------------------------------------------------------------
		checkCMAC();

		log.writeln( "Test AES     : OK\n" );
	}

	// -----------------------------------------------------------------------------------------------------------------
	// SP 800-38B - Recommendation for Block Cipher Modes of Operation: The CMAC Mode for Authentication
	// https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/AES_CMAC.pdf
	void checkCMAC()
	{
		//----------------------------------------------------------------------------------
		String key128 = "2B7E1516 28AED2A6 ABF71588 09CF4F3C";
		// Example #1
		AES alg = new AES();
		checkCMAC( alg, key128, testData.first(  0 ), null, "BB1D6929 E9593728 7FA37D12 9B756746" );
		// Example #2
		checkCMAC( alg, key128, testData.first( 16 ), null, "070A16B4 6B4D4144 F79BDD9D D04A287C" );
		// Example #3
		checkCMAC( alg, key128, testData.first( 20 ), null, "7D85449E A6EA19C8 23A7BF78 837DFADE" );
		// Example #4
		checkCMAC( alg, key128, testData,             null, "51F0BEBF 7E3B9D92 FC497417 79363CFE" );

		//----------------------------------------------------------------------------------
		String key192 = "8E73B0F7 DA0E6452 C810F32B 809079E5 62F8EAD2 522C6B7B";
		// Example #1
		checkCMAC( alg, key192, testData.first(  0 ), null, "D17DDF46 ADAACDE5 31CAC483 DE7A9367" );
		// Example #2
		checkCMAC( alg, key192, testData.first( 16 ), null, "9E99A7BF 31E71090 0662F65E 617C5184" );
		// Example #3
		checkCMAC( alg, key192, testData.first( 20 ), null, "3D75C194 ED960704 44A9FA7E C740ECF8" );
		// Example #4
		checkCMAC( alg, key192, testData,             null, "A1D5DF0E ED790F79 4D775896 59F39A11" );

		//----------------------------------------------------------------------------------
		String key256 = "603DEB10 15CA71BE 2B73AEF0 857D7781 1F352C07 3B6108D7 2D9810A3 0914DFF4";

		// Example #1
		checkCMAC( alg, key256, testData.first(  0 ), null, "028962F6 1B7BF89E FC6B551F 4667D983" );
		// Example #2
		checkCMAC( alg, key256, testData.first( 16 ), null, "28A7023F 452E8F82 BD4BF28D 8C37C35C" );
		// Example #3
		checkCMAC( alg, key256, testData.first( 20 ), null, "156727DC 0878944A 023C1FE0 3BAD6D93" );
		// Example #4
		checkCMAC( alg, key256, testData,             null, "E1992190 549F6ED5 696A2C05 6C315410" );
	}

}