// Denom.org
// bouncycastle.org

package org.denom.crypt.blockcipher;

import org.denom.Binary;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;
import static org.denom.crypt.blockcipher.Camellia128.*;

/**
 * Camellia - RFC 3713.
 * keySize = 24 or 32 bytes.
 */
public class Camellia256 extends BlockCipher
{
	private static final int BLOCK_SIZE = 16;

	private int[] subkeyE = new int[ 24 * 4 ];
	private int[] kwE = new int[ 4 * 2 ]; // for whitening
	private int[] keE = new int[ 6 * 2 ]; // for FL and FL^(-1)

	private int[] subkeyD = new int[ 24 * 4 ];
	private int[] kwD = new int[ 4 * 2 ]; // for whitening
	private int[] keD = new int[ 6 * 2 ]; // for FL and FL^(-1)

	private int[] state = new int[ 4 ];

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public Camellia256()
	{
		this( Bin( 32 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 bytes]
	 */
	public Camellia256( final Binary key )
	{
		super.initialize( BLOCK_SIZE );
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Camellia256 clone()
	{
		return new Camellia256( this.key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "Camellia256";
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Generate random key. Generated key will be set as current.
	 * @param keySize - 24 or 32 bytes.
	 */
	public Binary generateKey( int keySize )
	{
		MUST( (keySize == 24) || (keySize == 32), "Invalid key size" );
		Binary akey = new Binary().randomSecure( keySize );
		setKey( akey );
		return akey;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary generateKey()
	{
		return generateKey( 32 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [24 or 32 bytes].
	 */
	@Override
	public void setKey( Binary key )
	{
		MUST( (key.size() == 24) || (key.size() == 32), "Invalid key size" );
		this.key = key.clone();

		// forEncryption
		int[] k = new int[ 8 ];
		int[] ka = new int[ 4 ];
		int[] kb = new int[ 4 ];

		k[ 0 ] = key.getIntBE( 0 );
		k[ 1 ] = key.getIntBE( 4 );
		k[ 2 ] = key.getIntBE( 8 );
		k[ 3 ] = key.getIntBE( 12 );
		k[ 4 ] = key.getIntBE( 16 );
		k[ 5 ] = key.getIntBE( 20 );
		if( key.size() == 24 )
		{
			k[ 6 ] = ~k[ 4 ];
			k[ 7 ] = ~k[ 5 ];
		}
		else
		{
			k[ 6 ] = key.getIntBE( 24 );
			k[ 7 ] = key.getIntBE( 28 );
		}

		for( int i = 0; i < 4; i++ )
			ka[ i ] = k[ i ] ^ k[ i + 4 ];

		camelliaF2( ka, SIGMA, 0 );
		for( int i = 0; i < 4; i++ )
			ka[ i ] ^= k[ i ];

		camelliaF2( ka, SIGMA, 4 );

		for( int i = 0; i < 4; i++ )
			kb[ i ] = ka[ i ] ^ k[ i + 4 ];

		camelliaF2( kb, SIGMA, 8 );

		// KL dependant keys
		kwE[ 0 ] = k[ 0 ];
		kwE[ 1 ] = k[ 1 ];
		kwE[ 2 ] = k[ 2 ];
		kwE[ 3 ] = k[ 3 ];
		roldqo32( 45, k, 0, subkeyE, 16 );
		roldq( 15, k, 0, keE, 4 );
		roldq( 17, k, 0, subkeyE, 32 );
		roldqo32( 34, k, 0, subkeyE, 44 );
		// KR dependant keys
		roldq( 15, k, 4, subkeyE, 4 );
		roldq( 15, k, 4, keE, 0 );
		roldq( 30, k, 4, subkeyE, 24 );
		roldqo32( 34, k, 4, subkeyE, 36 );
		// KA dependant keys
		roldq( 15, ka, 0, subkeyE, 8 );
		roldq( 30, ka, 0, subkeyE, 20 );
		// 32bit rotation
		keE[ 8 ] = ka[ 1 ];
		keE[ 9 ] = ka[ 2 ];
		keE[ 10 ] = ka[ 3 ];
		keE[ 11 ] = ka[ 0 ];
		roldqo32( 49, ka, 0, subkeyE, 40 );

		// KB dependant keys
		subkeyE[ 0 ] = kb[ 0 ];
		subkeyE[ 1 ] = kb[ 1 ];
		subkeyE[ 2 ] = kb[ 2 ];
		subkeyE[ 3 ] = kb[ 3 ];
		roldq( 30, kb, 0, subkeyE, 12 );
		roldq( 30, kb, 0, subkeyE, 28 );
		roldqo32( 51, kb, 0, kwE, 4 );


		// for decrypt
		k = new int[ 8 ];
		ka = new int[ 4 ];
		kb = new int[ 4 ];

		k[ 0 ] = key.getIntBE( 0 );
		k[ 1 ] = key.getIntBE( 4 );
		k[ 2 ] = key.getIntBE( 8 );
		k[ 3 ] = key.getIntBE( 12 );
		k[ 4 ] = key.getIntBE( 16 );
		k[ 5 ] = key.getIntBE( 20 );
		if( key.size() == 24 )
		{
			k[ 6 ] = ~k[ 4 ];
			k[ 7 ] = ~k[ 5 ];
		}
		else
		{
			k[ 6 ] = key.getIntBE( 24 );
			k[ 7 ] = key.getIntBE( 28 );
		}

		for( int i = 0; i < 4; i++ )
			ka[ i ] = k[ i ] ^ k[ i + 4 ];

		camelliaF2( ka, SIGMA, 0 );
		for( int i = 0; i < 4; i++ )
			ka[ i ] ^= k[ i ];

		camelliaF2( ka, SIGMA, 4 );

		for( int i = 0; i < 4; i++ )
			kb[ i ] = ka[ i ] ^ k[ i + 4 ];

		camelliaF2( kb, SIGMA, 8 );

		// KL dependant keys
		kwD[ 4 ] = k[ 0 ];
		kwD[ 5 ] = k[ 1 ];
		kwD[ 6 ] = k[ 2 ];
		kwD[ 7 ] = k[ 3 ];
		decroldqo32( 45, k, 0, subkeyD, 28 );
		decroldq( 15, k, 0, keD, 4 );
		decroldq( 17, k, 0, subkeyD, 12 );
		decroldqo32( 34, k, 0, subkeyD, 0 );
		// KR dependant keys
		decroldq( 15, k, 4, subkeyD, 40 );
		decroldq( 15, k, 4, keD, 8 );
		decroldq( 30, k, 4, subkeyD, 20 );
		decroldqo32( 34, k, 4, subkeyD, 8 );
		// KA dependant keys
		decroldq( 15, ka, 0, subkeyD, 36 );
		decroldq( 30, ka, 0, subkeyD, 24 );
		// 32bit rotation
		keD[ 2 ] = ka[ 1 ];
		keD[ 3 ] = ka[ 2 ];
		keD[ 0 ] = ka[ 3 ];
		keD[ 1 ] = ka[ 0 ];
		decroldqo32( 49, ka, 0, subkeyD, 4 );

		// KB dependant keys
		subkeyD[ 46 ] = kb[ 0 ];
		subkeyD[ 47 ] = kb[ 1 ];
		subkeyD[ 44 ] = kb[ 2 ];
		subkeyD[ 45 ] = kb[ 3 ];
		decroldq( 30, kb, 0, subkeyD, 32 );
		decroldq( 30, kb, 0, subkeyD, 16 );
		roldqo32( 51, kb, 0, kwD, 0 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void processBlock( Binary block, int[] subkey, int[] kw, int ke[] )
	{
		for( int i = 0; i < 4; i++ )
		{
			state[ i ] = block.getIntBE( i * 4 );
			state[ i ] ^= kw[ i ];
		}

		camelliaF2( state, subkey, 0 );
		camelliaF2( state, subkey, 4 );
		camelliaF2( state, subkey, 8 );
		camelliaFLs( state, ke, 0 );
		camelliaF2( state, subkey, 12 );
		camelliaF2( state, subkey, 16 );
		camelliaF2( state, subkey, 20 );
		camelliaFLs( state, ke, 4 );
		camelliaF2( state, subkey, 24 );
		camelliaF2( state, subkey, 28 );
		camelliaF2( state, subkey, 32 );
		camelliaFLs( state, ke, 8 );
		camelliaF2( state, subkey, 36 );
		camelliaF2( state, subkey, 40 );
		camelliaF2( state, subkey, 44 );

		block.setIntBE( 0, state[ 2 ] ^ kw[ 4 ] );
		block.setIntBE( 4, state[ 3 ] ^ kw[ 5 ] );
		block.setIntBE( 8, state[ 0 ] ^ kw[ 6 ] );
		block.setIntBE( 12, state[ 1 ] ^ kw[ 7 ] );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void encryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );
		processBlock( block, subkeyE, kwE, keE );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void decryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );
		processBlock( block, subkeyD, kwD, keD );
	}

}
