// Denom.org
// bouncycastle.org

package org.denom.crypt.blockcipher;

import org.denom.Binary;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

/**
 * A class that provides a basic International Data Encryption Algorithm (IDEA) engine.
 *
 * This implementation is based on the "HOWTO: INTERNATIONAL DATA ENCRYPTION ALGORITHM"
 * implementation summary by Fauzan Mirza (F.U.Mirza@sheffield.ac.uk). (barring 1 typo at the
 * end of the mulinv function!).
 *
 * It can be found at ftp://ftp.funet.fi/pub/crypt/cryptography/symmetric/idea/
 *
 * Note: This algorithm was patented in the USA, Japan and Europe. These patents expired in 2011/2012. 
 */
public class IDEA extends BlockCipher
{
	public static final int BLOCK_SIZE = 8;
	public static final int KEY_SIZE = 16;

	private int[] keyE = null;
	private int[] keyD = null;

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public IDEA()
	{
		this( Bin( KEY_SIZE ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 bytes]
	 */
	public IDEA( final Binary key )
	{
		super.initialize( BLOCK_SIZE );
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public IDEA clone()
	{
		return new IDEA( this.key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "IDEA";
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary generateKey()
	{
		Binary k = new Binary().randomSecure( KEY_SIZE );
		setKey( k );
		return k;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void setKey( Binary key )
	{
		MUST( key.size() == KEY_SIZE, "Invalid key size" );
		this.key = key.clone();

		expandKey();
		invertKey();
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void encryptBlock( Binary block )
	{
		processBlock( keyE, block );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void decryptBlock( Binary block )
	{
		processBlock( keyD, block );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * return x = x * y where the multiplication is done modulo 65537 (0x10001) (as defined in the IDEA specification)
	 * and a zero input is taken to be 65536 (0x10000).
	 *
	 * @param x the x value
	 * @param y the y value
	 * @return x = x * y
	 */
	private int mul( int x, int y )
	{
		if( x == 0 )
		{
			x = (0x10001 - y);
		}
		else if( y == 0 )
		{
			x = (0x10001 - x);
		}
		else
		{
			int p = x * y;

			y = p & 0xffff;
			x = p >>> 16;
			x = y - x + ((y < x) ? 1 : 0);
		}

		return x & 0xffff;
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void processBlock( int[] wKey, Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		int x0 = block.getU16( 0 );
		int x1 = block.getU16( 2 );
		int x2 = block.getU16( 4 );
		int x3 = block.getU16( 6 );

		int keyOff = 0;
		for( int round = 0; round < 8; round++ )
		{
			x0 = mul( x0, wKey[ keyOff++ ] );
			x1 += wKey[ keyOff++ ];
			x1 &= 0xffff;
			x2 += wKey[ keyOff++ ];
			x2 &= 0xffff;
			x3 = mul( x3, wKey[ keyOff++ ] );

			int t0 = x1;
			int t1 = x2;
			x2 ^= x0;
			x1 ^= x3;

			x2 = mul( x2, wKey[ keyOff++ ] );
			x1 += x2;
			x1 &= 0xffff;

			x1 = mul( x1, wKey[ keyOff++ ] );
			x2 += x1;
			x2 &= 0xffff;

			x0 ^= x1;
			x3 ^= x2;
			x1 ^= t1;
			x2 ^= t0;
		}

		block.setU16( 0, mul( x0, wKey[ keyOff++ ] ) );
		block.setU16( 2, x2 + wKey[ keyOff++ ] );
		block.setU16( 4, x1 + wKey[ keyOff++ ] );
		block.setU16( 6,  mul( x3, wKey[ keyOff ] ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Expand the user key to the encryption subkey.
	 * The first 16 bytes are the user key, and the rest of the subkey is calculated by rotating the previous 16 bytes
	 * by 25 bits to the left, and so on until the subkey is completed.
	 */
	private void expandKey()
	{
		keyE = new int[ 52 ];

		for( int i = 0; i < 8; i++ )
			keyE[ i ] = key.getU16( i * 2 );

		for( int i = 8; i < 52; i++ )
		{
			if( (i & 7) < 6 )
			{
				keyE[ i ] = ((keyE[ i - 7 ] & 127) << 9 | keyE[ i - 6 ] >> 7) & 0xffff;
			}
			else if( (i & 7) == 6 )
			{
				keyE[ i ] = ((keyE[ i - 7 ] & 127) << 9 | keyE[ i - 14 ] >> 7) & 0xffff;
			}
			else
			{
				keyE[ i ] = ((keyE[ i - 15 ] & 127) << 9 | keyE[ i - 14 ] >> 7) & 0xffff;
			}
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Computes multiplicative inverse using Euclid's Greatest Common Divisor algorithm. Zero and one are self inverse.
	 * i.e. x * mulInv(x) == 1 (modulo BASE)
	 */
	private int mulInv( int x )
	{
		int t0, t1, q, y;

		if( x < 2 )
		{
			return x;
		}

		t0 = 1;
		t1 = 0x10001 / x;
		y = 0x10001 % x;

		while( y != 1 )
		{
			q = x / y;
			x = x % y;
			t0 = (t0 + (t1 * q)) & 0xffff;
			if( x == 1 )
			{
				return t0;
			}
			q = y / x;
			y = y % x;
			t1 = (t1 + (t0 * q)) & 0xffff;
		}

		return (1 - t1) & 0xffff;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Return the additive inverse of x.
	 * i.e. x + addInv(x) == 0
	 */
	private int addInv( int x )
	{
		return (0 - x) & 0xffff;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Invert the encryption subkey to the decryption subkey. It also involves the
	 * multiplicative inverse and the additive inverse functions.
	 */
	private void invertKey()
	{
		int t1, t2, t3, t4;
		int p = 52; // We work backwards
		keyD = new int[ 52 ];
		int inOff = 0;

		t1 = mulInv( keyE[ inOff++ ] );
		t2 = addInv( keyE[ inOff++ ] );
		t3 = addInv( keyE[ inOff++ ] );
		t4 = mulInv( keyE[ inOff++ ] );
		keyD[ --p ] = t4;
		keyD[ --p ] = t3;
		keyD[ --p ] = t2;
		keyD[ --p ] = t1;

		for( int round = 1; round < 8; round++ )
		{
			t1 = keyE[ inOff++ ];
			t2 = keyE[ inOff++ ];
			keyD[ --p ] = t2;
			keyD[ --p ] = t1;

			t1 = mulInv( keyE[ inOff++ ] );
			t2 = addInv( keyE[ inOff++ ] );
			t3 = addInv( keyE[ inOff++ ] );
			t4 = mulInv( keyE[ inOff++ ] );
			keyD[ --p ] = t4;
			keyD[ --p ] = t2; // NB: Order
			keyD[ --p ] = t3;
			keyD[ --p ] = t1;
		}

		t1 = keyE[ inOff++ ];
		t2 = keyE[ inOff++ ];
		keyD[ --p ] = t2;
		keyD[ --p ] = t1;

		t1 = mulInv( keyE[ inOff++ ] );
		t2 = addInv( keyE[ inOff++ ] );
		t3 = addInv( keyE[ inOff++ ] );
		t4 = mulInv( keyE[ inOff ] );
		keyD[ --p ] = t4;
		keyD[ --p ] = t3;
		keyD[ --p ] = t2;
		keyD[ --p ] = t1;
	}
}
