// Denom.org
// bouncycastle.org

package org.denom.crypt.blockcipher;

import org.denom.Binary;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

import static java.lang.Long.rotateLeft;
import static java.lang.Long.rotateRight;

/**
 * The specification for RC5 came from the 'RC5 Encryption Algorithm' publication in RSA CryptoBytes, Spring of 1995.
 * http://www.rsasecurity.com/rsalabs/cryptobytes
 *
 * This implementation is set to work with a 64 bit word size.
 * Implementation courtesy of Tito Pena.
 */
public class RC5_64 extends BlockCipher
{
	private static final int BLOCK_SIZE = 16;

	/**
	 * Pw = Odd((e-2) * 2^wordsize)
	 * Qw = Odd((o-2) * 2^wordsize)
	 * where e is the base of natural logarithms (2.718281828...) and o is the golden ratio (1.61803398...)
	 */
	private static final long P64 = 0xb7e151628aed2a6bL;
	private static final long Q64 = 0x9e3779b97f4a7c15L;

	private int roundsNum;
	private long[] sKey;

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public RC5_64()
	{
		this( 12, Bin( 16 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public RC5_64( int roundsNumber )
	{
		this( roundsNumber, Bin( 16 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public RC5_64( int roundsNumber, final Binary key )
	{
		super.initialize( BLOCK_SIZE );
		roundsNum = roundsNumber;
		sKey = new long[ 2 * (roundsNum + 1) ];
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public RC5_64 clone()
	{
		return new RC5_64( roundsNum, this.key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "RC5-64";
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key - 1 <= keySize <= 255.
	 */
	@Override
	public void setKey( final Binary key )
	{
		MUST( (key.size() >= 1) && (key.size() <= 255), "Invalid key size" );
		this.key = key.clone();

		// KEY EXPANSION:
		// There are 3 phases to the key expansion.
		// Phase 1:
		//   Copy the secret key K[0...b-1] into an array L[0..c-1] of
		//   c = ceil(b/u), where u = wordSize/8 in little-endian order.
		//   In other words, we fill up L using u consecutive key bytes
		//   of K. Any unfilled byte positions in L are zeroed. In the
		//   case that b = c = 0, set c = 1 and L[0] = 0.
		long[] L = new long[ (key.size() + (8 - 1)) / 8 ];

		for( int i = 0; i != key.size(); i++ )
		{
			L[ i / 8 ] += (long)key.get( i ) << (8 * (i % 8));
		}

		// Phase 2:
		//   Initialize S to a particular fixed pseudo-random bit pattern
		//   using an arithmetic progression modulo 2^wordsize determined
		//   by the magic numbers, Pw & Qw.
		sKey[ 0 ] = P64;
		for( int i = 1; i < sKey.length; i++ )
		{
			sKey[ i ] = (sKey[ i - 1 ] + Q64);
		}

		// Phase 3:
		//   Mix in the user's secret key in 3 passes over the arrays S & L.
		//   The max of the arrays sizes is used as the loop control
		int iter;
		if( L.length > sKey.length )
		{
			iter = 3 * L.length;
		}
		else
		{
			iter = 3 * sKey.length;
		}

		long A = 0, B = 0;
		int i = 0, j = 0;
		for( int k = 0; k < iter; k++ )
		{
			A = sKey[ i ] = rotateLeft( sKey[ i ] + A + B, 3 );
			B = L[ j ] = rotateLeft( L[ j ] + A + B, (int)(A + B) );
			i = (i + 1) % sKey.length;
			j = (j + 1) % L.length;
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Generate random key. Generated key will be set as current.
	 * @param keySize - 1 <= sz <= 255.
	 */
	public Binary generateKey( int keySize )
	{
		MUST( (keySize >= 1) && (keySize <= 255), "Invalid key size" );
		Binary akey = new Binary().randomSecure( keySize );
		setKey( akey );
		return akey;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary generateKey()
	{
		Binary k = new Binary().randomSecure( 16 );
		setKey( k );
		return k;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void encryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		long A = block.getLongLE( 0 ) + sKey[ 0 ];
		long B = block.getLongLE( 8 ) + sKey[ 1 ];

		for( int i = 1; i <= roundsNum; i++ )
		{
			A = rotateLeft( A ^ B, (int)B ) + sKey[ 2 * i ];
			B = rotateLeft( B ^ A, (int)A ) + sKey[ 2 * i + 1 ];
		}

		block.setLongLE( 0, A );
		block.setLongLE( 8, B );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void decryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		long A = block.getLongLE( 0 );
		long B = block.getLongLE( 8 );

		for( int i = roundsNum; i >= 1; i-- )
		{
			B = rotateRight( B - sKey[ 2 * i + 1 ], (int)A ) ^ A;
			A = rotateRight( A - sKey[ 2 * i ], (int)B ) ^ B;
		}

		block.setLongLE( 0, A - sKey[ 0 ] );
		block.setLongLE( 8, B - sKey[ 1 ] );
	}

}
