// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.crypt.blockcipher;

import org.denom.*;

import static org.denom.Binary.*;
import static org.denom.Ex.*;

/**
 * Block cipher TripleDES: DES-EDE2 (if key size = 16 bytes) or DES-EDE3 (if key size = 24 bytes).
 */
public class TripleDES extends BlockCipher
{
	public static final int BLOCK_SIZE = 8;

	private DES des1 = new DES();
	private DES des2 = new DES();
	private DES des3 = new DES();

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public TripleDES()
	{
		this( Bin( 16 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 or 24 bytes]
	 */
	public TripleDES( final Binary key )
	{
		super.initialize( BLOCK_SIZE );
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public TripleDES( String keyHex )
	{
		this( Bin( keyHex ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public TripleDES clone()
	{
		return new TripleDES( this.key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "TripleDES";
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 or 24 bytes].
	 */
	@Override
	public void setKey( final Binary key )
	{
		MUST( (key.size() == 16) || (key.size() == 24), "Invalid TripleDES key size" );
		this.key = key.clone();

		des1.setKey( key.first( 8 ) );
		des2.setKey( key.slice( 8, 8 ) );
		if( key.size() == 24 )
			des3.setKey( key.last( 8 ) );
		else
			des3.setKey( key.first( 8 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Generate random key. Generated key will be set as current.
	 */
	@Override
	public Binary generateKey()
	{
		return generateKey( key.size() );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Generate random key. Generated key will be set as current.
	 * @param keySize - 16 or 24 bytes.
	 */
	public Binary generateKey( int keySize )
	{
		MUST( (keySize == 16) || (keySize == 24), "Invalid TripleDES key size" );
		Binary akey = new Binary().randomSecure( keySize );
		DES.setOddParityBits( akey );
		setKey( akey );
		return akey;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void encryptBlock( Binary block )
	{
		des1.encryptBlock( block );
		des2.decryptBlock( block );
		des3.encryptBlock( block );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void decryptBlock( Binary block )
	{
		des3.decryptBlock( block );
		des2.encryptBlock( block );
		des1.decryptBlock( block );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * ISO 9797-1, algorithm 3
	 * @return mac
	 */
	public Binary calcMACAlg3( final Binary data, AlignMode alignMode, final Binary iv )
	{
		Binary mac = des1.calcMAC( data, alignMode, iv );
		des2.decryptBlock( mac );
		des3.encryptBlock( mac );
		return mac;
	}

	// ---------------------------------------------------------------------------------------------------------------------
	/**
	 * Вычислить контрольное значение для ключа.
	 * @return контрольное значение [3 байта].
	 */
	public Binary calcKCV()
	{
		Binary crypt = encrypt( Bin( 8 ), CryptoMode.ECB, AlignMode.NONE, null );
		return crypt.slice( 0, 3 );
	}

}
