// Denom.org
// bouncycastle.org

package org.denom.crypt.streamcipher;

import org.denom.Binary;

import static org.denom.Ex.MUST;

/**
 * Daniel J. Bernstein's ChaCha stream cipher.
 * Key size = 32 bytes.
 * IV size = 12 bytes.
 */
public class ChaCha7539 extends Salsa20
{
	// -----------------------------------------------------------------------------------------------------------------
	public ChaCha7539()
	{
		super();
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param rounds the number of rounds (must be an even number).
	 */
	public ChaCha7539( int rounds )
	{
		super( rounds );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param rounds the number of rounds (must be an even number).
	 * @param key [32 bytes]
	 */
	public ChaCha7539( int rounds, final Binary key )
	{
		super( rounds, key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "ChaCha7539-" + rounds;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [32 bytes]
	 */
	@Override
	public void setKey( final Binary key )
	{
		MUST( key.size() == 32, "Wrong key size" );
		this.key = key.clone();
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param IV - 12 bytes.
	 */
	@Override
	public ChaCha7539 startEncrypt( final Binary iv )
	{
		MUST( (iv != null) && (iv.size() == 12), "Wrong IV size" );
		initChaCha7539( iv );
		reset();
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param IV - 8 bytes.
	 */
	@Override
	public ChaCha7539 startDecrypt( final Binary iv )
	{
		MUST( (iv != null) && (iv.size() == 12), "Wrong IV size" );
		initChaCha7539( iv );
		reset();
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void initChaCha7539( final Binary iv )
	{
		packTauOrSigma( key.size(), engineState, 0 );

		littleEndianToInt( key.getDataRef(), 0, engineState, 4, 8 );

		littleEndianToInt( iv.getDataRef(), 0, engineState, 13, 3 );
	}
	// -----------------------------------------------------------------------------------------------------------------
	protected void advanceCounter( long diff )
	{
		int hi = (int)(diff >>> 32);
		int lo = (int)diff;

		MUST( hi == 0, "attempt to increase counter past 2^32." );

		int oldState = engineState[ 12 ];

		engineState[ 12 ] += lo;

		if( oldState != 0 && engineState[ 12 ] < oldState )
		{
			throw new IllegalStateException( "attempt to increase counter past 2^32." );
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	protected void advanceCounter()
	{
		MUST( ++engineState[ 12 ] != 0, "attempt to increase counter past 2^32." );
	}

	// -----------------------------------------------------------------------------------------------------------------
	protected void retreatCounter( long diff )
	{
		int hi = (int)(diff >>> 32);
		int lo = (int)diff;

		if( hi != 0 )
		{
			throw new IllegalStateException( "attempt to reduce counter past zero." );
		}

		if( (engineState[ 12 ] & 0xffffffffL) >= (lo & 0xffffffffL) )
		{
			engineState[ 12 ] -= lo;
		}
		else
		{
			throw new IllegalStateException( "attempt to reduce counter past zero." );
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	protected void retreatCounter()
	{
		if( engineState[ 12 ] == 0 )
		{
			throw new IllegalStateException( "attempt to reduce counter past zero." );
		}

		--engineState[ 12 ];
	}

	// -----------------------------------------------------------------------------------------------------------------
	protected long getCounter()
	{
		return engineState[ 12 ] & 0xffffffffL;
	}

	// -----------------------------------------------------------------------------------------------------------------
	protected void resetCounter()
	{
		engineState[ 12 ] = 0;
	}

	// -----------------------------------------------------------------------------------------------------------------
	protected void generateKeyStream( byte[] output )
	{
		ChaCha.chachaCore( rounds, engineState, x );
		intToLittleEndian( x, output, 0 );
	}
}
