// Denom.org
// bouncycastle.org

package org.denom.crypt.streamcipher;

import java.util.Arrays;
import org.denom.Binary;

import static java.lang.Integer.rotateRight;
import static java.lang.Integer.rotateLeft;
import static org.denom.Binary.*;
import static org.denom.Ex.MUST;

/**
 * HC-256 - stream cipher created by Hongjun Wu.
 * It is a third phase candidate in the eStream contest, and is patent-free.
 */
public class HC128 extends StreamCipher
{
	private int[] p = new int[ 512 ];
	private int[] q = new int[ 512 ];
	private int cnt = 0;

	private byte[] buf = new byte[ 4 ];
	private int idx = 0;

	int[] w = new int[ 1280 ];

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public HC128()
	{
		this( Bin( 16 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 bytes]
	 */
	public HC128( final Binary key )
	{
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "HC-128";
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key - 16 bytes.
	 */
	@Override
	public void setKey( final Binary key )
	{
		MUST( key.size() == 16, "Wrong key size" );
		this.key = key.clone();
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param IV - 16 bytes.
	 */
	@Override
	public HC128 startEncrypt( final Binary iv )
	{
		init( iv );
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param IV - 16 bytes.
	 */
	@Override
	public HC128 startDecrypt( final Binary iv )
	{
		init( iv );
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void init( final Binary iv )
	{
		MUST( (iv != null) && iv.size() == 16, "Wrong IV size" );

		idx = 0;
		cnt = 0;

		Arrays.fill( w, 0 );

		for( int i = 0; i < 16; i++ )
		{
			w[ i >> 2 ] |= key.get( i ) << (8 * (i & 0x3));
		}
		System.arraycopy( w, 0, w, 4, 4 );

		for( int i = 0; i < iv.size() && i < 16; i++ )
		{
			w[ (i >> 2) + 8 ] |= iv.get( i ) << (8 * (i & 0x3));
		}
		System.arraycopy( w, 8, w, 12, 4 );

		for( int i = 16; i < 1280; i++ )
		{
			int x = w[ i - 2 ];
			int y = w[ i - 15 ];
			w[ i ] = (rotateRight( x, 17 ) ^ rotateRight( x, 19 ) ^ (x >>> 10)) + w[ i - 7 ] + (rotateRight( y, 7 ) ^ rotateRight( y, 18 ) ^ (y >>> 3)) + w[ i - 16 ] + i;
		}

		System.arraycopy( w, 256, p, 0, 512 );
		System.arraycopy( w, 768, q, 0, 512 );

		for( int i = 0; i < 512; i++ )
			p[ i ] = step();

		for( int i = 0; i < 512; i++ )
			q[ i ] = step();

		cnt = 0;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public byte process( byte in )
	{
		return (byte)(in ^ getByte());
	}

	// -----------------------------------------------------------------------------------------------------------------
	private byte getByte()
	{
		if( idx == 0 )
			setIntLE( buf, 0, step() );

		byte ret = buf[ idx ];
		idx = (idx + 1) & 0x03;
		return ret;
	}

	// -----------------------------------------------------------------------------------------------------------------
	private int step()
	{
		int j = mod512( cnt );
		int ret;
		if( cnt < 512 )
		{
			int x = p[ mod512( j-3 ) ];
			int y = p[ mod512( j-10 ) ];
			int z = p[ mod512( j-511 ) ];
			p[ j ] += (rotateRight( x, 10 ) ^ rotateRight( z, 23 )) + rotateRight( y, 8 );
			ret = h1( p[ mod512( j-12 ) ] ) ^ p[ j ];
		}
		else
		{
			int x = q[ mod512( j-3 ) ];
			int y = q[ mod512( j-10 ) ];
			int z = q[ mod512( j-511 ) ];
			q[ j ] += (rotateLeft( x, 10 ) ^ rotateLeft( z, 23 )) + rotateLeft( y, 8 );
			ret = h2( q[ mod512( j-12 ) ] ) ^ q[ j ];
		}

		cnt = (cnt + 1) & 0x3FF;
		return ret;
	}

	// -----------------------------------------------------------------------------------------------------------------
	private int h1( int x )
	{
		return q[ x & 0xFF ] + q[ ((x >> 16) & 0xFF) + 256 ];
	}

	// -----------------------------------------------------------------------------------------------------------------
	private int h2( int x )
	{
		return p[ x & 0xFF ] + p[ ((x >> 16) & 0xFF) + 256 ];
	}

	// -----------------------------------------------------------------------------------------------------------------
	private static int mod512( int x )
	{
		return x & 0x1FF;
	}
}
