// Denom.org
// bouncycastle.org

package org.denom.crypt.streamcipher;

import java.util.Arrays;
import org.denom.Binary;

import static org.denom.Binary.*;
import static org.denom.Ex.MUST;

/**
 * Bob Jenkin's ISAAC (Indirection Shift Accumulate Add and Count).
 * http://www.burtleburtle.net/bob/rand/isaacafa.html
 */
public class ISAAC extends StreamCipher
{
	private int[] state = new int[ 256 ];
	private int[] results = new int[ 256 ];
	private byte[] keyStream = new byte[ 1024 ];

	private int index = 0;
	private int a = 0;
	private int b = 0;
	private int c = 0;

	private byte[] tKey;

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public ISAAC()
	{
		this( Bin( 16 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [16 bytes]
	 */
	public ISAAC( final Binary key )
	{
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "ISAAC";
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param 1 <= Key size <= 1024 bytes.
	 */
	@Override
	public void setKey( final Binary key )
	{
		MUST( (key.size() >= 1) && (key.size() <= 1024), "Wrong key size" );
		this.key = key.clone();

		tKey = new byte[ (key.size() + 3) & ~3 ];
		System.arraycopy( key.getDataRef(), 0, tKey, 0, key.size() );

		init();
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void init()
	{
		Arrays.fill( state, 0 );
		a = 0;
		b = 0;
		c = 0;
		index = 0;

		Arrays.fill( results, 0 );
		for( int i = 0; i < tKey.length; i += 4 )
			results[ i >>> 2 ] = getIntLE( tKey, i );

		int[] abcdefgh = new int[ 8 ];

		for( int i = 0; i < 8; i++ )
			abcdefgh[ i ] = 0x9e3779b9; // Phi (golden ratio)

		for( int i = 0; i < 4; i++ )
			mix( abcdefgh );

		for( int i = 0; i < 2; i++ )
		{
			for( int j = 0; j < state.length; j += 8 )
			{
				for( int k = 0; k < 8; k++ )
				{
					abcdefgh[ k ] += (i < 1) ? results[ j + k ] : state[ j + k ];
				}

				mix( abcdefgh );

				for( int k = 0; k < 8; k++ )
				{
					state[ j + k ] = abcdefgh[ k ];
				}
			}
		}

		isaac();
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * IV is not used for this algorithm.
	 */
	@Override
	public ISAAC startEncrypt( final Binary iv )
	{
		init();
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * IV is not used for this algorithm.
	 */
	@Override
	public ISAAC startDecrypt( final Binary iv )
	{
		init();
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public byte process( byte in )
	{
		if( index == 0 )
		{
			isaac();
			for( int i = 0; i < results.length; ++i )
				setIntBE( keyStream, i << 2, results[ i ] );
		}
		byte out = (byte)(keyStream[ index ] ^ in);
		index = (index + 1) & 1023;

		return out;
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void isaac()
	{
		++c;
		b += c;

		for( int i = 0; i < results.length; i++ )
		{
			int x = state[ i ];
			switch( i & 3 )
			{
				case 0:
					a ^= (a << 13);
					break;
				case 1:
					a ^= (a >>> 6);
					break;
				case 2:
					a ^= (a << 2);
					break;
				case 3:
					a ^= (a >>> 16);
					break;
			}
			a += state[ (i + 128) & 0xFF ];
			int y = state[ (x >>> 2) & 0xFF ] + a + b;
			state[ i ] = y;
			b = state[ (y >>> 10) & 0xFF ] + x;
			results[ i ] = b;
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void mix( int[] x )
	{
		x[0] ^= x[1] <<  11;  x[3] += x[0];  x[1] += x[2];
		x[1] ^= x[2] >>>  2;  x[4] += x[1];  x[2] += x[3];
		x[2] ^= x[3] <<   8;  x[5] += x[2];  x[3] += x[4];
		x[3] ^= x[4] >>> 16;  x[6] += x[3];  x[4] += x[5];
		x[4] ^= x[5] <<  10;  x[7] += x[4];  x[5] += x[6];
		x[5] ^= x[6] >>>  4;  x[0] += x[5];  x[6] += x[7];
		x[6] ^= x[7] <<   8;  x[1] += x[6];  x[7] += x[0];
		x[7] ^= x[0] >>>  9;  x[2] += x[7];  x[0] += x[1];
	}
}
