package org.denom.format;

import java.nio.charset.StandardCharsets;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Map;
import java.util.Iterator;
import java.util.LinkedHashMap;

import org.denom.Ex;

import static org.denom.Ex.MUST;

/**
 * Список key-value, сериализуемый в формат JO.
 */
public class Jo
{
	public Map<String, Object> map;

	public Map<String, String> comments;

	// -----------------------------------------------------------------------------------------------------------------
	private static final char KEYVAL_DELIMITER = '=';
	private static final char ELEMS_DELIMITER = ';';

	// -----------------------------------------------------------------------------------------------------------------
	public Jo()
	{
		map = new LinkedHashMap<>();
	}

	// -----------------------------------------------------------------------------------------------------------------
	public int length()
	{
		return map.size();
	}

	// -----------------------------------------------------------------------------------------------------------------
	public Jo put( String key, double val )
	{
		put( key, Double.valueOf( val ) );
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	public Jo put( String key, Object val )
	{
		map.put( key, val );
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	static void appendIndent( Appendable writer, int indent ) throws IOException
	{
		while( indent > 0 )
		{
			writer.append( ' ' );
			--indent;
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Сериализация без переводов строк и лишних пробелов
	 * @param writer
	 */
	public void writeCompact( Appendable writer )
	{
		try
		{
			writer.append( '{' );

			Iterator< Map.Entry<String, Object> > it = map.entrySet().iterator();
			while( it.hasNext() )
			{
				Map.Entry<String, Object> entry = it.next();
				writer.append( entry.getKey() );
				writer.append( KEYVAL_DELIMITER );

				Object val = entry.getValue();

				if( val == null )
				{
					if( it.hasNext() )
						writer.append( ELEMS_DELIMITER );
				}
				else
				{
					if( val instanceof Jo )
					{
						((Jo)val).writeCompact( writer );
					}
					else
					{
						writer.append( val.toString() );

						if( it.hasNext() )
							writer.append( ELEMS_DELIMITER );
					}
				}
			}

			writer.append( '}' );
		}
		catch( IOException exception )
		{
			throw new Ex( exception );
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Сериализация в текстовый документ с переводами строк.
	 * @param indentFactor - Для каждого вложенного объекта будет добавляться по столько пробелов * уровень вложенности.
	 */
	public void write( Appendable writer, int indentFactor, int indent )
	{
		MUST( (indentFactor >= 0) && (indent >= 0), "Wrong indent params" );

		try
		{
			Iterator< Map.Entry<String, Object> > it = map.entrySet().iterator();
			while( it.hasNext() )
			{
				Map.Entry<String, Object> entry = it.next();
				Object val = entry.getValue();

				appendIndent( writer, indent );
				writer.append( entry.getKey() );
				writer.append( ' ' );
				writer.append( KEYVAL_DELIMITER );
				writer.append( ' ' );

				if( val == null )
				{
					writer.append( ELEMS_DELIMITER );
				}
				else
				{
					if( val instanceof Jo )
					{
						if( ((Jo)val).map.size() != 0 )
						{
							writer.append( '\n' );
							appendIndent( writer, indent );
							writer.append( '{' );
							writer.append( '\n' );
							((Jo)val).write( writer, indentFactor, indent + indentFactor );
							appendIndent( writer, indent );
						}
						else
						{
							writer.append( '{' );
						}
						writer.append( '}' );
					}
					else
					{
						writer.append( val.toString() );
					}
				}
				writer.append( '\n' );
			}
		}
		catch( IOException exception )
		{
			throw new Ex( exception );
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	public void save( String filename, int indentFactor, int indent )
	{
		try( BufferedWriter fw = Files.newBufferedWriter( Paths.get( filename ), StandardCharsets.UTF_8 ) )
		{
			 write( fw, indentFactor, indent );
		}
		catch( IOException ex )
		{
			throw new Ex( ex );
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	public String toString()
	{
		return toString( 4 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public String toString( int indentFactor )
	{
		StringBuilder sb = new StringBuilder( 64 );
		write( sb, indentFactor, 0 );
		return sb.toString();
	}
}
