// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard.cap;

import java.util.ArrayList;
import org.denom.Binary;
import org.denom.smartcard.cap.ComponentClass.TypeDescriptor;
import org.denom.smartcard.cap.ComponentConstantPool.ClassRef;
import org.denom.smartcard.cap.ComponentConstantPool.StaticRef;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

/**
 * CAP-file component - Descriptor.
 */
public class ComponentDescriptor extends CapComponent
{
	public int tag;
	public int size;

	public ClassDescriptorInfo[] classes;

	// types:  type descriptor info
	public int[] constPoolTypes;
	public ArrayList< TypeDescriptor > typeDesc;

	// -----------------------------------------------------------------------------------------------------------------
	public ComponentDescriptor( CapFile cap, String fullName, final Binary b )
	{
		super( cap, TAG_DESCRIPTOR, fullName );

		int offs = 0;

		tag = b.get( offs++ );
		size = b.getU16( offs ); offs += 2;

		int classCount = b.get( offs++ );
		classes = new ClassDescriptorInfo[ classCount ];
		for( int i = 0; i < classes.length; ++i )
		{
			classes[ i ] = new ClassDescriptorInfo();
			offs = classes[ i ].parse( b, offs );
		}

		// types -- type_descriptor_info

		int constPoolCount = b.getU16( offs ); offs += 2;
		constPoolTypes = new int[ constPoolCount ];
		for( int i = 0; i < constPoolTypes.length; ++i )
		{
			constPoolTypes[ i ] = b.getU16( offs );
			offs += 2;
		}

		typeDesc = new ArrayList< TypeDescriptor >();
		while( offs < b.size() )
		{
			TypeDescriptor td = new TypeDescriptor();
			offs = td.parse( b, offs );
			typeDesc.add( td );
		}

		MUST( (tag == TAG_DESCRIPTOR) && (size == (b.size() - 3))
				&& (offs == b.size()), "Wrong component Descriptor" );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public Binary toBin()
	{
		Binary b = Bin().reserve( size + 3 );

		b.add( tag );
		b.addU16( size );

		b.add( classes.length );
		for( ClassDescriptorInfo cl : classes )
			cl.toBin( b );

		// types -- type_descriptor_info
		b.addU16( constPoolTypes.length );
		for( int elem : constPoolTypes )
			b.addU16( elem );

		for( TypeDescriptor td : typeDesc )
			td.toBin( b );

		return b;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Найти информацию о методе, описание которого начинается по смещению 'methodOffset' в компоненте Method.
	 * Линейный поиск среди всех методов всех классов.
	 * @return null если метод не найден.
	 */
	public MethodDescriptorInfo findMethod( int methodOffset )
	{
		for( ClassDescriptorInfo cl : classes )
			for( MethodDescriptorInfo md : cl.methods )
				if( md.methodOffset == methodOffset )
					return md;
		return null;
	}

	// =================================================================================================================
	public static class FieldDescriptorInfo
	{
		public static final byte ACC_PUBLIC    = 0x01;
		public static final byte ACC_PRIVATE   = 0x02;
		public static final byte ACC_PROTECTED = 0x04;
		public static final byte ACC_STATIC    = 0x08;
		public static final byte ACC_FINAL     = 0x10;

		public int token;
		public int accessFlags;

		public StaticRef staticField;

		// instance field
		public ClassRef classRef;
		public int tokenRef;

		public int type;

		protected int parse( Binary b, int offs )
		{
			token = b.get( offs++ );
			accessFlags = b.get( offs++ );

			if( (accessFlags & ACC_STATIC) != 0 )
			{
				staticField = new StaticRef();
				offs = staticField.parse( b, offs );
			}
			else
			{
				classRef = new ClassRef();
				offs = classRef.parse( b, offs );
				tokenRef = b.get( offs++ );
			}

			type = b.getU16( offs ); offs += 2;

			return offs;
		}

		protected void toBin( Binary b )
		{
			b.add( token );
			b.add( accessFlags );

			if( (accessFlags & ACC_STATIC) != 0 )
			{
				staticField.toBin( b );
			}
			else
			{
				classRef.toBin( b );
				b.add( token );
			}

			b.addU16( type );
		}
	}

	// =================================================================================================================
	public static class MethodDescriptorInfo
	{
		public static final int ACC_PUBLIC    = 0x01;
		public static final int ACC_PRIVATE   = 0x02;
		public static final int ACC_PROTECTED = 0x04;
		public static final int ACC_STATIC    = 0x08;
		public static final int ACC_FINAL     = 0x10;
		public static final int ACC_ABSTRACT  = 0x40;
		public static final int ACC_INIT      = 0x80;

		public int token;
		public int accessFlags;
		public int methodOffset;
		public int typeOffset;
		public int bytecodeCount;
		public int exceptionHandlerCount;
		public int exceptionHandlerIndex;

		protected int parse( Binary b, int offs )
		{
			token = b.get( offs++ );
			accessFlags = b.get( offs++ );
			methodOffset = b.getU16( offs ); offs += 2;
			typeOffset = b.getU16( offs ); offs += 2;
			bytecodeCount = b.getU16( offs ); offs += 2;
			exceptionHandlerCount = b.getU16( offs ); offs += 2;
			exceptionHandlerIndex = b.getU16( offs ); offs += 2;
			return offs;
		}

		protected void toBin( Binary b )
		{
			b.add( token );
			b.add( accessFlags );
			b.addU16( methodOffset );
			b.addU16( typeOffset );
			b.addU16( bytecodeCount );
			b.addU16( exceptionHandlerCount );
			b.addU16( exceptionHandlerIndex );
		}
	}

	// =================================================================================================================
	public static class ClassDescriptorInfo
	{
		public static final int ACC_PUBLIC    = 0x01;
		public static final int ACC_FINAL     = 0x10;
		public static final int ACC_INTERFACE = 0x40;
		public static final int ACC_ABSTRACT  = 0x80;

		public int token;
		public int accessFlags;
		public ClassRef thisClassRef;
		public ClassRef[] interfaces;
		public FieldDescriptorInfo[] fields;
		public MethodDescriptorInfo[] methods;

		protected int parse( Binary b, int offs )
		{
			token = b.get( offs++ );
			accessFlags = b.get( offs++ );

			thisClassRef = new ClassRef();
			offs = thisClassRef.parse( b, offs );

			int interfaceCount = b.get( offs++ );
			int fieldCount = b.getU16( offs ); offs += 2;
			int methodCount = b.getU16( offs ); offs += 2;

			interfaces = new ClassRef[ interfaceCount ];
			for( int i = 0; i < interfaceCount; ++i )
			{
				interfaces[ i ] = new ClassRef();
				offs = interfaces[ i ].parse( b, offs );
			}

			fields = new FieldDescriptorInfo[ fieldCount ];
			for( int i = 0; i < fieldCount; ++i )
			{
				fields[ i ] = new FieldDescriptorInfo();
				offs = fields[ i ].parse( b, offs );
			}

			methods = new MethodDescriptorInfo[ methodCount ];
			for( int i = 0; i < methodCount; ++i )
			{
				methods[ i ] = new MethodDescriptorInfo();
				offs = methods[ i ].parse( b, offs );
			}

			return offs;
		}

		protected void toBin( Binary b )
		{
			b.add( token );
			b.add( accessFlags );

			thisClassRef.toBin( b );

			b.add( interfaces.length );
			b.addU16( fields.length );
			b.addU16( methods.length );

			for( ClassRef intf : interfaces )
				intf.toBin( b );

			for( FieldDescriptorInfo field : fields )
				field.toBin( b );

			for( MethodDescriptorInfo method : methods )
				method.toBin( b );
		}
	}

}
