// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard.emv;

import org.denom.Binary;
import org.denom.Ex;
import org.denom.crypt.blockcipher.*;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

/**
 * Криптографические примитивы для EMV-систем, работающих по контактному интерфейсу.
 * Алгоритмы описаны в EMV 4.4, Book 2:
 *   EMV Integrated Circuit Card Specifications for Payment Systems.
 *   Book 2. Security and Key Management. Version 4.4, October 2022.
 */
public class EmvCrypt
{
	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычисление TripleDES ключа карты (MK) из мастер-ключа эмитента (IMK).
	 * @param IMK - TripleDES-ключ эмитента [16 байт].
	 * @param Y - [8 байт], данные для получения производного ключа: 7 последних байт PAN + PAN SN.
	 * @return Производный ключ.
	 */
	public static Binary deriveMKTripleDes( final TripleDES IMK, final Binary Y )
	{
		MUST( IMK.getKeySize() == 16, "Wrong key size" );
		MUST( Y.size() == 8, "Wrong size of derivation data" );

		Binary Y2 = Binary.xor( Y, Bin( 8, 0xFF ) );
		Binary Z = Bin( Y, Y2 );
		Z = IMK.encrypt( Z, CryptoMode.ECB, AlignMode.NONE, null );
		DES.setOddParityBits( Z );
		return Z;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычисление TripleDES ключа карты (MK) из мастер-ключа эмитента (IMK).
	 * EMV 4.4, Book 2, A1.4.1  Option A
	 * Book E, 4.1.A
	 */
	public static Binary deriveMKTripleDes( final TripleDES IMK, final Binary PAN, final Binary PAN_SN )
	{
		// PAN || PSN
		Binary X = Bin( PAN, PAN_SN );
		// Rightmost 16 digits of X
		Binary Y = X.last( 8 );
		return deriveMKTripleDes( IMK, Y );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Генерация AES ключа карты (MK) из мастер-ключа эмитента (IMK).
	 * EMV 4.4, Book 2, A1.4.3  Option C
	 * Book E, 4.1.C
	 * @return MK. Размер равен размеру ключа IMK, т.е. 16 / 24 / 32 байта.
	 */
	public static Binary deriveMKAes( final AES IMK, final Binary PAN, final Binary PAN_SN )
	{
		// X = PAN || PSN
		// Pad X to the left with zero-digits to form a 16-byte number Y
		Binary Y = Bin( Bin( 16 - PAN.size() - PAN_SN.size() ), PAN, PAN_SN );

		int keySize = IMK.getKeySize();
		if( keySize == 16 )
		{
			IMK.encryptBlock( Y );
			return Y;
		}
		else
		{
			Binary Y2 = Binary.xor( Y, Bin( 16, 0xFF ) );
			Binary MK = IMK.encrypt( Bin( Y, Y2 ), CryptoMode.ECB, AlignMode.NONE, null );
			return MK.first( keySize );
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Calc EMV Session Key.
	 * @param MK - card key - TripleDES or AES
	 * @param R - данные для диверсификации [blockSize байт].
	 * @return session key - размер равен размеру MK.
	 */
	public static Binary calcSK( final BlockCipher MK, final Binary R )
	{
		MUST( R.size() == MK.getBlockSize(), "Wrong R size for EMV session key" );

		if( (MK instanceof AES) && (MK.getKeySize() == 16) )
		{
			Binary sk = R.clone();
			MK.encryptBlock( sk );
			return sk;
		}
		else
		{
			Binary l = R.clone();
			l.set( 2, 0xF0 );
			Binary r = R.clone();
			r.set( 2, 0x0F );

			Binary sk = MK.encrypt( Bin( l, r ), CryptoMode.ECB, AlignMode.NONE, null ).first( MK.getKeySize() );
			return sk;
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Calc card session key for AC.
	 * EMV 4.4, Book2, A1.3.1 Common Session Key Derivation Option.
	 * Этот же алгоритм описан для бесконтакта в Book E, 4.2  Application Cryptogram.
	 * @param MKac - Ключ карты для вычисления криптограмм DES2_EDE или AES.
	 * @param atc - Счётчик криптограмм карты (EMV тег - 0x9F36) [2 байта].
	 * @return SKac
	 */
	public static Binary calcSKac( final BlockCipher MKac, final Binary atc )
	{
		MUST( atc.size() == 2, "Wrong ATC size" );
		Binary R = atc.clone();
		R.resize( MKac.getBlockSize() );
		Binary SKac = calcSK( MKac, R );
		return SKac;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Сгенерировать сессионный ключ TripleDES SKac для вычисления криптограмм карты (AC).
	 * Для Mastercard - отличается от стандартного EMV.
	 * @param kdm - Алгоритм вывода сессионного ключа, см. SKDerivationMethod.
	 * @param MKac - Ключ карты для вычисления криптограмм [16 байт].
	 * @param atc - Счётчик криптограмм карты (EMV тег - 0x9F36) [2 байта].
	 * @param unpredictableNumber - Случайное число терминала [4 байта],
	 * не используется в EMV, нужно в MChip. (EMV тег - 0x9F37) ( default = Bin() ).
	 * @return [16 байт] Сессионный ключ SKac. 
	 */
	public static Binary calcTripleDesSKac( int kdm, TripleDES MKac, final Binary atc, 
		final Binary unpredictableNumber )
	{
		if( kdm == SKDerivationMethod.MCHIP )
		{
			MUST( atc.size() == 2, "Wrong ATC size" );
			MUST( unpredictableNumber.size() == 4, "Wrong unpredictableNumber size" );

			Binary R = atc.clone();
			R.resize( 4 );
			R.add( unpredictableNumber );
			Binary SKac = calcSK( MKac, R );
			return SKac;
		}
		else
		{
			return calcSKac( MKac, atc );
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * EMV 4.4, Book 2, A1.2.1. MAC Algorithms Using an 8-byte Block Cipher.
	 * MAC для ключей TripleDES считается по ISO 9797-1, algorithm 3.
	 * Урезание результата до меньшего количества байт можно сделать снаружи.
	 * @param msg - произвольные данные, выравниваются до кратности размеру блока внутри функции.
	 * @return MAC [8 байт]
	 */
	public static Binary calcMAC( TripleDES SK, final Binary msg )
	{
		return SK.calcMACAlg3( msg, AlignMode.BLOCK, null );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * EMV 4.4, Book 2, A1.2.2. MAC Algorithms Using an 16-byte Block Cipher.
	 * MAC для ключей AES - это CMAC.
	 * Урезание результата до меньшего количества байт можно сделать снаружи.
	 * @param msg - произвольные данные, выравниваются до кратности размеру блока внутри функции.
	 * @return MAC [8 байт]
	 */
	public static Binary calcMAC( AES SK, final Binary msg )
	{
		return SK.calcCMAC( msg, null ).first( 8 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Calc card AC for TripleDES alg.
	 * EMV 4.4, Book 2. 8.1.2  Application Cryptogram Algorithm.
	 * Этот же алгоритм описан для бесконтакта в Book E, 4.2  Application Cryptogram
	 * @return AC [8 байт]
	 */
	public static Binary calcAC( TripleDES SKac, final Binary msg )
	{
		return calcMAC( SKac, msg );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Calc card AC for AES alg.
	 * EMV 4.4, Book 2. 8.1.2  Application Cryptogram Algorithm.
	 * Этот же алгоритм описан для бесконтакта в Book E, 4.2  Application Cryptogram
	 * @return AC [8 байт]
	 */
	public static Binary calcAC( AES SKac, final Binary msg )
	{
		return calcMAC( SKac, msg );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычисление ARPC Method 1 для TripleDES или AES.
	 * EMV 4.4, Book 2. 8.2.1  ARPC Method 1.
	 * @param ARQC - криптограмма карты (AC) [8 байт].
	 * @param ARC - Authorisation Response Code (ARC) [2 байта]
	 * @return ARPC [8 байт].
	 */
	public static Binary calcARPCMethod1( BlockCipher SKac, final Binary ARQC, final Binary ARC )
	{
		MUST( ARC.size() == 2, "Wrong ARC size" );
		MUST( ARQC.size() == 8, "Wrong ARQC size" );

		Binary Y = Bin( 8 );
		Y.set( 0, ARC, 0, 2 );
		Y.xor( ARQC );
		Y.resize( SKac.getBlockSize() );
		SKac.encryptBlock( Y );
		return Y;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычисление ARPC Method 2 для TripleDES или AES.
	 * EMV 4.4, Book 2. 8.2.2  ARPC Method 2.
	 * @param ARQC - криптограмма карты (AC) [8 байт].
	 * @param CSU - Card Status Update [4 байта].
	 * @param proprietaryAuthData - Proprietary Authentication Data [0-8 байт], может быть null.
	 * @return ARPC [4 байта]
	 */
	public static Binary calcARPCMethod2( BlockCipher SKac, final Binary ARQC, final Binary CSU, final Binary proprietaryAuthData )
	{
		MUST( ARQC.size() == 8, "Wrong ARQC size" );
		MUST( CSU.size() == 4, "Wrong CSU size" );
		
		Binary Y = ARQC.clone();
		Y.add( CSU );
		if( proprietaryAuthData != null )
		{
			MUST( proprietaryAuthData.size() <= 8, "Wrong proprietaryAuthData size" );
			Y.add( proprietaryAuthData );
		}

		if( SKac instanceof TripleDES )
		{
			return calcMAC( (TripleDES)SKac, Y ).first( 4 );
		}
		else if( SKac instanceof AES )
		{
			return calcMAC( (AES)SKac, Y ).first( 4 );
		}
		else
		{
			throw new Ex( "Wrong SKac algorithm" );
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычисление Issuer Authentication Data (tag '91'), ARPC Method 2 для TripleDES или AES.
	 * EMV 4.4, Book 2. 8.2.2  ARPC Method 2.
	 * см. комментарии к методу calcARPCMethod2.
	 * @return Issuer Authentication Data [4 + 4 + proprietaryAuthData.size()] -- value для TLV-объекта с тегом 91.
	 */
	public static Binary calcIssuerAuthData( BlockCipher SKac, final Binary ARQC, final Binary CSU, final Binary proprietaryAuthData )
	{
		// Calc ARPC
		Binary res = calcARPCMethod2( SKac, ARQC, CSU, proprietaryAuthData );

		res.add( CSU );

		if( proprietaryAuthData != null )
			res.add( proprietaryAuthData );

		return res;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычислить сессионный ключ SKsmi для Secure Messaging Integrity.
	 * @param MKsmi - ключ карты.
	 * @param ac - криптограмма карты.
	 * @return
	 */
	public static AES calcSKsmi( AES MKsmi, Binary ac )
	{
		Binary b = Bin( ac, Bin( 8 ) );
		MKsmi.encryptBlock( b );
		return new AES( b );
	}
}
