// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard.emv.kernel8;

import org.denom.Binary;

/**
 * Интерфейс для класса, вычисляющего криптограммы во время сессии с картой по спецификации для Kernel 8.
 * Алгоритмы вычисления криптограмм зависят от Secure Channel Algorithm Suite,
 * выбранному терминалом из поддерживаемых картой.
 */
public interface IKernel8Crypter
{
	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Возвращает Secure Channel Algorithm Suite Indicator, для которого реализован класс.
	 * Например: 0x00, 0x88, 0x8D.
	 */
	int getASI();

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Проверить, поддерживает ли карта ASI, реализованный классом - getASI().
	 * @param cardQualifier - 8 байт. Байты 2-4 - список алгоритмов, поддерживаемых картой. 
	 */
	public default boolean isASISupported( final Binary cardQualifier )
	{
		int myASI = getASI();

		for( int i = 1; i <= 3; ++i )
			if( cardQualifier.get( i ) == myASI )
				return true;

		return false;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Сбросить все сессионные данные.
	 * Инициализировать их начальными значениями для новой сессии.
	 */
	void resetSession();

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @return KernelQualifier для передачи карте
	 */
	Binary getKernelQualifier();

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Генерирует value для TLV-объекта 'Kernel Key Data' (Tag = 9E).
	 * Для передачи карте в GPO.
	 */
	Binary getKernelKeyData();

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Обработать ключевые данные карты:  Pc(X) || E(r).
	 * @param cardKeyData - value TLV-объекта с тегом 9F8103, которые карта возвращает в GPO.
	 * @return false - if data is incorrect.
	 */
	boolean processCardKeyData( final Binary cardKeyData );

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычислить SDA Hash перед проверкой сертификатов или вычислением IAD-MAC.
	 */
	void calcSdaHash( final Binary sdaData );
	
	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Проверить сертификаты карты.
	 * @return false - if data is incorrect.
	 */
	boolean processCertificates( final Binary caPKIndex, final Binary caPublicKey,
			final Binary issuerCert, final Binary iccCert, final Binary aid, final Binary pan );

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычислить IAD-MAC на сессионных ключах.
	 */
	Binary calcIADMac( final Binary PDOLValues, final Binary CDOL1RelatedData,
			final Binary terminalRREntropy, final Binary lastERRDResponse, final Binary genAcResponse );

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычислить EDA-MAC на сессионных ключах.
	 */
	Binary calcEDAMac( final Binary IADMac, final Binary cardAC );

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param rec - TLV with tag 0x70 or 0xDA (encrypted).
	 * @return Decrypted record (TLV with tag 0x70).
	 */
	Binary decryptRecord( final Binary rec );

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Зашифровать/Расшифровать Data Envelope для отправки в карту в команде WRITE DATA.
	 */
	Binary cryptWriteData( final Binary plainTlv );

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычислить MAC от Plain TLV  Data Envelope для ответа карты на WRITE DATA
	 */
	Binary calcDataEnvelopeMac( final Binary plainTlv );
	
	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Расшифровать ответ карты на READ DATA.
	 * @return Plain TLV или null, если MAC не совпал.
	 */
	Binary decryptReadData( final Binary crypt );

}
