// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard.emv.kernel8;

import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Random;

import org.denom.Binary;
import org.denom.format.BerTLV;
import org.denom.crypt.ec.ECAlg;
import org.denom.crypt.ec.Fp.custom.Secp256r1;
import org.denom.crypt.blockcipher.AES;
import org.denom.crypt.hash.SHA256;
import org.denom.smartcard.emv.TagEmv;
import org.denom.smartcard.emv.certificate.CertificateEccNISTIcc;
import org.denom.smartcard.emv.certificate.CertificateEccNISTIssuer;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

/**
 * Вычисляет криптограммы во время сессии с картой по спецификации для Kernel 8.
 * Secure Channel Algorithm Suite = 0x00 (P-256, AES-CMAC, AES-CTR)
 */
public class Kernel8CrypterNIST implements IKernel8Crypter
{
	public static int ASI = 0x00;

	public static int ECNField = 32;
	public static Binary kernelQualifier = Bin( "02 80 00 10 FF FF 00 00" );


	private SHA256 hashAlg = new SHA256();

	private ECAlg ecAlg = new ECAlg( new Secp256r1(), new Random( System.nanoTime() ) );

	// Terminal ephimeral key pair.
	private ECAlg ecDk = ecAlg.clone();

	// Blinded card public key
	private final Binary Pc = Bin();
	private final Binary rRecovered = Bin();

	private Binary sdaHash = Bin();

	private final AES SKc = new AES();
	private final AES SKi = new AES();

	// Card Message Counter
	private int CMC = 0x8000;
	// Kernel Message Counter
	private int KMC = 0x0000;

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Возвращает Algorithm Suite Indicator, для которого реализован класс.
	 * Например: 0x00, 0x88, 0x8D
	 */
	@Override
	public int getASI()
	{
		return ASI;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary getKernelQualifier()
	{
		return kernelQualifier.clone();
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void resetSession()
	{
		Pc.clear();
		rRecovered.clear();
		sdaHash.clear();
		SKc.setKey( Bin(16) );
		SKi.setKey( Bin(16) );
		CMC = 0x8000;
		KMC = 0x0000;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary getKernelKeyData()
	{
		ecDk.generateKeyPair();

		Binary Qk = ecDk.getPublic( false );
		// Без первого байта, только  X || Y точки.
		return Qk.last( Qk.size() - 1 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public boolean processCardKeyData( final Binary cardKeyData )
	{
		if( cardKeyData.size() != (ECNField * 2) )
			return false;

		try
		{
			Pc.assign( cardKeyData, 0, ECNField );

			Binary[] SKcSKi = Kernel8Crypt.BDHCalcSessionKeys( ecDk, Pc );
			SKc.setKey( SKcSKi[ 0 ] );
			SKi.setKey( SKcSKi[ 1 ] );

			Binary encryptedR = cardKeyData.last( ECNField );
			rRecovered.assign( Kernel8Crypt.BDHRecoverR( SKc, encryptedR, CMC ) );
			CMC++;
		}
		catch( Throwable ex )
		{
			return false;
		}

		return true;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * в ecAlg - ключ CA.
	 * @return в ecAlg - EC-ключ Issuer-а.
	 */
	private boolean processIssuerCertificate( final Binary caPKIndex, final Binary issuerCertBin, final Binary aid, final Binary pan )
	{
		try
		{
			// Проверки по Book-E, 5.2 Issuer ECC Public Key Validation

			// При парсинге сертификата проверяются п. 1, 2, 3, 5, 10
			CertificateEccNISTIssuer issuerCert = new CertificateEccNISTIssuer().fromBin( issuerCertBin );

			// 4. Check that the Issuer Identifier matches the leftmost 3-10 digits from the Application PAN obtained from the Card.
			String panStr = pan.Hex( 0 );
			String issuerIdStr = issuerCert.issuerId.Hex( 0 );
			// Сколько значащих цифр в issuerId
			int lenIId = issuerIdStr.indexOf( 'F' );
			if( lenIId == -1 )
				lenIId = issuerIdStr.length();

			panStr = panStr.substring( 0, lenIId );
			issuerIdStr = issuerIdStr.substring( 0, lenIId );
			if( !panStr.equals( issuerIdStr ) )
				return false;

			// 6. Check that the Certificate Expiration Date is equal to or later than the current date.
			Binary now = Bin( ZonedDateTime.now( ZoneOffset.UTC ).format( DateTimeFormatter.ofPattern("yyyyMMdd") ) );
			if( issuerCert.expirationDate.compareTo( now ) < 0 )
				return false;

			// 7. Check that the RID matches the RID in the first 5 bytes of AID obtained from the Card (DF Name).
			if( !issuerCert.RID.equals( aid.first( 5 ) ) )
				return false;

			// 8. Check that the Certification Authority Public Key Index obtained from the Card is the same as the one in Table 5.1.
			if( !issuerCert.caPKIndex.equals( caPKIndex ) )
				return false;

			// 9. Check that the concatenation of the RID, Certification Authority Public Key Index,
			// and Issuer Certificate Serial Number is not present on the Certification Revocation List described in Annex B.2.

			// No CRL in this Kernel8

			if( !issuerCert.verifySignature( ecAlg ) )
				return false;

			Kernel8Crypt.restorePublicKey( ecAlg, issuerCert.issuerPublicKeyX );

			return true;
		}
		catch( Throwable ex )
		{
			return false;
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * в ecAlg - ключ Issuer-а.
	 * @return в ecAlg - EC-ключ ICC.
	 */
	private boolean processICCCertificate( final Binary iccCertBin, final Binary sdaHash )
	{
		try
		{
			// Проверки по Book-E, 5.4  ICC ECC Public Key Validation

			// При парсинге сертификата проверяются п. 1, 2, 3, 5, 6, 7, 8

			CertificateEccNISTIcc iccCert = new CertificateEccNISTIcc().fromBin( iccCertBin );

			// 4. Check that the ICC Certificate Expiration Date and ICC Certificate Expiration Time
			// is equal to or later than the current date and time.
			Binary now = Bin( ZonedDateTime.now( ZoneOffset.UTC ).format( DateTimeFormatter.ofPattern("yyyyMMddHHmm") ) );
			Binary dateTime = Bin( iccCert.expirationDate, iccCert.expirationTime );
			if( dateTime.compareTo( now ) < 0 )
				return false;

			// 9. Check that the SDA Hash already computed by the Kernel is the same as the ICCD Hash recovered from the ICC certificate.
			if( !sdaHash.equals( iccCert.iccdHash ) )
				return false;

			// 10. Check that the ICC Public Key Certificate Signature is valid using the Issuer Public Key as described in section 8.8.7.
			if( !iccCert.verifySignature( ecAlg ) )
				return false;

			// Y - любой (не важен), т.к. ключ будет применяться только в ECDH для сравнения Pc (только X-координата).
			ecAlg.setPublic( Bin("02").add( iccCert.iccPublicKeyX ) );
		}
		catch( Throwable ex )
		{
			return false;
		}
		return true;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void calcSdaHash( final Binary sdaData )
	{
		sdaHash = hashAlg.calc( sdaData );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public boolean processCertificates( final Binary caPKIndex, final Binary caPublicKey,
			final Binary issuerCert, final Binary iccCert, final Binary aid, final Binary pan )
	{
		ecAlg.setPublic( caPublicKey );
		boolean isOk = processIssuerCertificate( caPKIndex, issuerCert, aid, pan );
		if( !isOk )
			return false;

		isOk = processICCCertificate( iccCert, sdaHash );
		if( !isOk )
			return false;

		if( !Kernel8Crypt.BDHValidate( ecAlg, rRecovered, Pc ) )
			return false;

		return true;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary calcIADMac( final Binary PDOLValues, final Binary CDOL1RelatedData,
			final Binary terminalRREntropy, final Binary lastERRDResponse,
			final Binary genAcResponse )
	{
		Binary data = Kernel8Crypt.formDataForIADMac( PDOLValues, CDOL1RelatedData, terminalRREntropy, lastERRDResponse,
				genAcResponse, sdaHash );
		return Kernel8Crypt.calcIADMac( SKi, data );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary calcEDAMac( final Binary IADMac, final Binary cardAC )
	{
		return Kernel8Crypt.calcEDAMac( SKi, cardAC, IADMac );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param rec - TLV with tag 0x70 or 0xDA (encrypted).
	 * @return Decrypted record (TLV with tag 0x70).
	 */
	@Override
	public Binary decryptRecord( final Binary rec )
	{
		BerTLV tlv = new BerTLV( rec );
		if( tlv.tag == TagEmv.ReadRecordResponseMessageTemplate )
			return rec.clone();

		MUST( tlv.tag == 0xDA, "Wrong tag in record" );
		Binary plain = Kernel8Crypt.cryptCTR( SKc, CMC++, tlv.value );
		Binary res = BerTLV.Tlv( TagEmv.ReadRecordResponseMessageTemplate, plain );
		return res;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary cryptWriteData( final Binary plainTlv )
	{
		return Kernel8Crypt.cryptCTR( SKc, KMC++, plainTlv );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary calcDataEnvelopeMac( final Binary plainTlv )
	{
		return Kernel8Crypt.calcMac( SKi, CMC++, plainTlv );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary decryptReadData( final Binary crypt )
	{
		return Kernel8Crypt.decryptReadData( SKc, SKi, CMC++, crypt );
	}

}
