/*
 * Decompiled with CFR 0.152.
 */
package org.denom.crypt.blockcipher;

import org.denom.Binary;
import org.denom.Ex;
import org.denom.crypt.blockcipher.AlignMode;
import org.denom.crypt.blockcipher.CryptoMode;

public abstract class BlockCipher {
    private int blockSize;
    protected Binary key;
    protected Binary iv0;
    private AlignMode tAlignMode;
    private CryptoMode tCryptMode;
    private Binary tIV;
    private Binary prevIV;
    private int tOperation = 0;
    private static final int OP_ENCRYPTION = 1;
    private static final int OP_DECRYPTION = 2;
    private static final int OP_MAC = 3;
    private Binary tail;
    private int N3 = 0;
    private int N4 = 0;
    private static final int GOST_C1 = 0x1010104;
    private static final int GOST_C2 = 0x1010101;

    public abstract BlockCipher clone();

    public abstract String getAlgName();

    public int getBlockSize() {
        return this.blockSize;
    }

    public int getKeySize() {
        return this.key.size();
    }

    public Binary getKey() {
        return this.key.clone();
    }

    public abstract Binary generateKey();

    public abstract void setKey(Binary var1);

    public void setKey(String keyHex) {
        this.setKey(Binary.Bin(keyHex));
    }

    protected void initialize(int blockSize) {
        this.blockSize = blockSize;
        this.iv0 = new Binary(blockSize);
        this.tIV = this.iv0.clone();
        this.prevIV = this.iv0.clone();
        this.tail = Binary.Bin().reserve(blockSize);
    }

    public abstract void encryptBlock(Binary var1);

    public abstract void decryptBlock(Binary var1);

    public Binary encrypt(Binary data, CryptoMode cryptMode, AlignMode alignMode, Binary iv) {
        this.startEncrypt(cryptMode, alignMode, iv);
        return this.finish(data);
    }

    public Binary decrypt(Binary data, CryptoMode cryptMode, AlignMode alignMode, Binary iv) {
        this.startDecrypt(cryptMode, alignMode, iv);
        return this.finish(data);
    }

    public Binary calcMAC(Binary data, AlignMode alignMode, Binary iv) {
        this.startMAC(alignMode, iv);
        return this.finish(data);
    }

    private void initIVAndTail(Binary iv) {
        if (iv != null) {
            Ex.MUST(iv.size() == this.blockSize, "Wrong IV size");
            this.tIV = iv.clone();
        } else {
            this.tIV = this.iv0.clone();
        }
        this.tail.resize(0);
    }

    public void startEncrypt(CryptoMode cryptMode, AlignMode alignMode, Binary iv) {
        this.tOperation = 1;
        this.tCryptMode = cryptMode;
        this.tAlignMode = alignMode;
        this.initIVAndTail(iv);
        if (cryptMode == CryptoMode.Gost28147CTR) {
            this.startGostCTR();
        }
    }

    private void startGostCTR() {
        Ex.MUST(this.blockSize == 8, "For GostOFB mode need BlockSize = 8");
        this.encryptBlock(this.tIV);
        this.N3 = this.tIV.getIntLE(0);
        this.N4 = this.tIV.getIntLE(4);
    }

    private void calcGostCTRGamma() {
        this.N3 += 0x1010101;
        this.N4 += 0x1010104;
        if (this.N4 < 0x1010104 && this.N4 > 0) {
            ++this.N4;
        }
        this.tIV.setIntLE(0, this.N3);
        this.tIV.setIntLE(4, this.N4);
        this.encryptBlock(this.tIV);
    }

    private void encryptBlockModes(Binary block) {
        switch (this.tCryptMode) {
            case ECB: {
                this.encryptBlock(block);
                break;
            }
            case CBC: {
                block.xor(this.tIV);
                this.encryptBlock(block);
                this.tIV.assign(block);
                break;
            }
            case CFB: {
                this.encryptBlock(this.tIV);
                block.xor(this.tIV);
                this.tIV.assign(block);
                break;
            }
            case OFB: {
                this.encryptBlock(this.tIV);
                block.xor(this.tIV);
                break;
            }
            case Gost28147CTR: {
                this.calcGostCTRGamma();
                block.xor(this.tIV);
            }
        }
    }

    private Binary updateEncrypt(Binary plain) {
        Binary ciphered = Binary.Bin();
        ciphered.reserve(plain.size() + (this.tail.size() > 0 ? this.blockSize : 0));
        int plainOffset = 0;
        int plainSize = plain.size();
        while (plainOffset < plainSize) {
            int sz = Math.min(this.blockSize - this.tail.size(), plain.size() - plainOffset);
            this.tail.add(plain, plainOffset, sz);
            plainOffset += sz;
            if (this.tail.size() != this.blockSize) continue;
            this.encryptBlockModes(this.tail);
            ciphered.add(this.tail);
            this.tail.clear();
        }
        return ciphered;
    }

    private Binary finishEncrypt(Binary plain) {
        Binary ciphered = this.updateEncrypt(plain);
        BlockCipher.pad(this.tail, this.blockSize, this.tAlignMode);
        if (this.tail.size() > 0) {
            if (this.tCryptMode == CryptoMode.ECB || this.tCryptMode == CryptoMode.CBC) {
                Ex.MUST(this.tail.size() == this.blockSize, "Wrong data size for encrypt");
            }
            this.encryptBlockModes(this.tail);
            ciphered.add(this.tail);
            this.tail.clear();
        }
        return ciphered;
    }

    public void startDecrypt(CryptoMode cryptMode, AlignMode alignMode, Binary iv) {
        this.tOperation = 2;
        this.tCryptMode = cryptMode;
        this.tAlignMode = alignMode;
        this.initIVAndTail(iv);
        if (cryptMode == CryptoMode.Gost28147CTR) {
            this.startGostCTR();
        }
    }

    private void decryptBlockModes(Binary block) {
        switch (this.tCryptMode) {
            case ECB: {
                this.decryptBlock(block);
                break;
            }
            case CBC: {
                this.prevIV.assign(block);
                this.decryptBlock(block);
                block.xor(this.tIV);
                this.tIV.assign(this.prevIV);
                break;
            }
            case CFB: {
                this.prevIV.assign(block);
                this.encryptBlock(this.tIV);
                block.xor(this.tIV);
                this.tIV.assign(this.prevIV);
                break;
            }
            case OFB: {
                this.encryptBlock(this.tIV);
                block.xor(this.tIV);
                break;
            }
            case Gost28147CTR: {
                this.calcGostCTRGamma();
                block.xor(this.tIV);
            }
        }
    }

    private Binary updateDecrypt(Binary ciphered) {
        Binary plain = Binary.Bin();
        plain.reserve(ciphered.size() + (this.tail.size() > 0 ? this.blockSize : 0));
        int cipheredOffset = 0;
        int cipheredSize = ciphered.size();
        while (cipheredOffset < cipheredSize) {
            int sz = Math.min(this.blockSize - this.tail.size(), ciphered.size() - cipheredOffset);
            this.tail.add(ciphered, cipheredOffset, sz);
            cipheredOffset += sz;
            if (this.tail.size() != this.blockSize) continue;
            this.decryptBlockModes(this.tail);
            plain.add(this.tail);
            this.tail.resize(0);
        }
        return plain;
    }

    private Binary finishDecrypt(Binary ciphered) {
        Binary plain = this.updateDecrypt(ciphered);
        if ((this.tCryptMode == CryptoMode.CFB || this.tCryptMode == CryptoMode.OFB || this.tCryptMode == CryptoMode.Gost28147CTR) && this.tAlignMode == AlignMode.NONE && this.tail.size() > 0) {
            this.decryptBlockModes(this.tail);
            plain.add(this.tail);
            this.tail.resize(0);
        }
        Ex.MUST(this.tail.size() == 0, "Wrong data size for decrypt");
        BlockCipher.unPad(plain, this.blockSize, this.tAlignMode);
        return plain;
    }

    public void startMAC(AlignMode alignMode, Binary iv) {
        this.tOperation = 3;
        this.tCryptMode = CryptoMode.CBC;
        this.tAlignMode = alignMode;
        this.initIVAndTail(iv);
    }

    private void updateMAC(Binary dataPart) {
        int plainOffset = 0;
        int plainSize = dataPart.size();
        while (plainOffset < plainSize) {
            int sz = Math.min(this.blockSize - this.tail.size(), dataPart.size() - plainOffset);
            this.tail.add(dataPart, plainOffset, sz);
            plainOffset += sz;
            if (this.tail.size() != this.blockSize) continue;
            this.tail.xor(this.tIV);
            this.encryptBlock(this.tail);
            this.tIV.assign(this.tail);
            this.tail.resize(0);
        }
    }

    private Binary finishMAC(Binary dataPart) {
        this.updateMAC(dataPart);
        BlockCipher.pad(this.tail, this.blockSize, this.tAlignMode);
        if (this.tail.size() > 0) {
            Ex.MUST(this.tail.size() == this.blockSize, "Wrong data size for MAC");
            this.tail.xor(this.tIV);
            this.encryptBlock(this.tail);
            this.tIV.assign(this.tail);
            this.tail.resize(0);
        }
        Ex.MUST(this.tail.size() == 0, "Wrong data size for MAC");
        return this.tIV.clone();
    }

    public Binary process(Binary dataPart) {
        switch (this.tOperation) {
            case 1: {
                return this.updateEncrypt(dataPart);
            }
            case 2: {
                return this.updateDecrypt(dataPart);
            }
            case 3: {
                this.updateMAC(dataPart);
                return null;
            }
        }
        throw new Ex("Crypt operation not started");
    }

    public Binary finish(Binary lastDataPart) {
        int op = this.tOperation;
        this.tOperation = 0;
        switch (op) {
            case 1: {
                return this.finishEncrypt(lastDataPart);
            }
            case 2: {
                return this.finishDecrypt(lastDataPart);
            }
            case 3: {
                return this.finishMAC(lastDataPart);
            }
        }
        throw new Ex("Crypt operation not started");
    }

    public Binary cryptCTR(Binary data, Binary SV, int JBytes) {
        Ex.MUST(JBytes > 0 && JBytes <= this.blockSize && SV.size() <= this.blockSize, "Wrong params for CTR");
        Binary Qi = SV.clone();
        Qi.resize(this.blockSize);
        Binary Ei = Binary.Bin(this.blockSize);
        Binary C = Binary.Bin(data.size());
        byte[] CArr = C.getDataRef();
        byte[] dataArr = data.getDataRef();
        byte[] EArr = Ei.getDataRef();
        int PSize = data.size();
        int offset = 0;
        while (offset < PSize) {
            Ei.assign(Qi);
            this.encryptBlock(Ei);
            int partSize = Math.min(JBytes, PSize - offset);
            int i = 0;
            while (i < partSize) {
                CArr[offset + i] = (byte)(dataArr[offset + i] ^ EArr[i]);
                ++i;
            }
            Qi.increment();
            offset += JBytes;
        }
        return C;
    }

    public Binary cryptCTR(Binary data, Binary SV) {
        return this.cryptCTR(data, SV, this.getBlockSize());
    }

    private void CMAC_SHL(Binary K) {
        byte[] k = K.getDataRef();
        int kSize = K.size();
        int nextBit = 0;
        int i = kSize - 1;
        while (i >= 0) {
            int tmp = k[i] & 0xFF;
            k[i] = (byte)(tmp << 1 | nextBit);
            nextBit = (byte)(tmp >> 7);
            --i;
        }
        int n = kSize - 1;
        k[n] = (byte)(k[n] ^ (byte)(0x87 & -nextBit));
    }

    public Binary calcCMAC(Binary data, Binary iv) {
        Binary res;
        if (iv != null) {
            Ex.MUST(iv.size() == this.blockSize, "Wrong IV length");
            res = iv.clone();
        } else {
            res = this.iv0.clone();
        }
        byte[] resArr = res.getDataRef();
        byte[] dataArr = data.getDataRef();
        int numDataBlocks = (data.size() + this.blockSize - 1) / this.blockSize;
        int offset = 0;
        int i = 1;
        while (i < numDataBlocks) {
            int j = 0;
            while (j < this.blockSize) {
                int n = j++;
                resArr[n] = (byte)(resArr[n] ^ dataArr[offset]);
                ++offset;
            }
            this.encryptBlock(res);
            ++i;
        }
        int restSize = data.size() - offset;
        Binary K = Binary.Bin(this.blockSize);
        this.encryptBlock(K);
        this.CMAC_SHL(K);
        int j = 0;
        while (j < restSize) {
            int n = j++;
            resArr[n] = (byte)(resArr[n] ^ dataArr[offset++]);
        }
        if (restSize != this.blockSize) {
            int n = j;
            resArr[n] = (byte)(resArr[n] ^ 0x80);
            this.CMAC_SHL(K);
        }
        res.xor(K);
        this.encryptBlock(res);
        return res;
    }

    public static void pad(Binary data, int blockSize, AlignMode alignMode) {
        switch (alignMode) {
            case NONE: {
                break;
            }
            case BLOCK: {
                Ex.MUST(blockSize > 0, "Wrong data alignment");
                int padLen = blockSize - (data.size() & blockSize - 1);
                data.reserve(data.size() + padLen);
                data.add(128);
                --padLen;
                int i = 0;
                while (i < padLen) {
                    data.add(0);
                    ++i;
                }
                break;
            }
            default: {
                throw new Ex("Wrong data alignment");
            }
        }
    }

    public static void unPad(Binary data, int blockSize, AlignMode alignMode) {
        switch (alignMode) {
            case NONE: {
                break;
            }
            case BLOCK: {
                Ex.MUST(blockSize > 0, "Wrong data alignment");
                Ex.MUST(!data.empty(), "Wrong data alignment");
                Ex.MUST((data.size() & blockSize - 1) == 0, "Wrong data alignment");
                int i = data.size();
                while (--i > 0 && data.get(i) == 0) {
                }
                Ex.MUST(data.get(i) == 128 && data.size() - i <= blockSize, "Wrong data alignment");
                data.resize(i);
                break;
            }
            default: {
                throw new Ex("Wrong data alignment");
            }
        }
    }
}

