// Denom.org
// bouncycastle.org

package org.denom.crypt.blockcipher;

import org.denom.Binary;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

import static java.lang.Integer.rotateLeft;
import static java.lang.Integer.rotateRight;

/**
 * The specification for RC5 came from the 'RC5 Encryption Algorithm' publication in RSA CryptoBytes, Spring of 1995. 
 * http://www.rsasecurity.com/rsalabs/cryptobytes
 * 
 * This implementation has a word size of 32 bits.
 * Implementation courtesy of Tito Pena.
 */
public class RC5_32 extends BlockCipher
{
	private static final int BLOCK_SIZE = 8;

	/**
	 * Pw = Odd((e-2) * 2^wordsize)
	 * Qw = Odd((o-2) * 2^wordsize)
	 * where e is the base of natural logarithms (2.718281828...) and o is the golden ratio (1.61803398...)
	 */
	private static final int P32 = 0xb7e15163;
	private static final int Q32 = 0x9e3779b9;

	private int roundsNum;
	private int[] sKey;

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Set key later.
	 */
	public RC5_32()
	{
		this( 12, Bin( 16 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public RC5_32( int roundsNumber )
	{
		this( roundsNumber, Bin( 16 ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public RC5_32( int roundsNumber, final Binary key )
	{
		super.initialize( BLOCK_SIZE );
		roundsNum = roundsNumber;
		sKey = new int[ 2 * (roundsNum + 1) ];
		setKey( key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public RC5_32 clone()
	{
		return new RC5_32( roundsNum, this.key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "RC5-32";
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key - 1 <= keySize <= 255.
	 */
	@Override
	public void setKey( final Binary key )
	{
		MUST( (key.size() >= 1) && (key.size() <= 255), "Invalid key size" );
		this.key = key.clone();

		// KEY EXPANSION:
		// Phase 1:
		//   Copy the secret key K[0...b-1] into an array L[0..c-1] of
		//   c = ceil(b/u), where u = 32/8 in little-endian order.
		//   In other words, we fill up L using u consecutive key bytes
		//   of K. Any unfilled byte positions in L are zeroed. In the
		//   case that b = c = 0, set c = 1 and L[0] = 0.
		int[] L = new int[ (key.size() + (4 - 1)) / 4 ];

		for( int i = 0; i != key.size(); i++ )
		{
			L[ i / 4 ] += key.get( i ) << (8 * (i % 4));
		}

		// Phase 2:
		//   Initialize S to a particular fixed pseudo-random bit pattern
		//   using an arithmetic progression modulo 2^wordsize determined by the magic numbers, Pw & Qw.
		sKey[ 0 ] = P32;
		for( int i = 1; i < sKey.length; ++i )
		{
			sKey[i] = sKey[i-1] + Q32;
		}

		// Phase 3:
		//   Mix in the user's secret key in 3 passes over the arrays S & L.
		//   The max of the arrays sizes is used as the loop control
		int iter;
		if( L.length > sKey.length )
		{
			iter = 3 * L.length;
		}
		else
		{
			iter = 3 * sKey.length;
		}

		int A = 0, B = 0;
		int i = 0, j = 0;

		for (int k = 0; k < iter; k++)
		{
			A = sKey[ i ] = rotateLeft( sKey[ i ] + A + B, 3 );
			B = L[ j ] = rotateLeft( L[ j ] + A + B, A + B );
			i = (i+1) % sKey.length;
			j = (j+1) %  L.length;
		}
	}
	
	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Generate random key. Generated key will be set as current.
	 * @param keySize - 1 <= sz <= 255.
	 */
	public Binary generateKey( int keySize )
	{
		MUST( (keySize >= 1) && (keySize <= 255), "Invalid key size" );
		Binary akey = new Binary().randomSecure( keySize );
		setKey( akey );
		return akey;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public Binary generateKey()
	{
		Binary k = new Binary().randomSecure( 16 );
		setKey( k );
		return k;
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void encryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		int A = block.getIntLE( 0 ) + sKey[ 0 ];
		int B = block.getIntLE( 4 ) + sKey[ 1 ];

		for (int i = 1; i <= roundsNum; i++)
		{
			A = rotateLeft( A ^ B, B ) + sKey[ 2 * i ];
			B = rotateLeft( B ^ A, A ) + sKey[ 2 * i + 1 ];
		}
		
		block.setIntLE( 0, A );
		block.setIntLE( 4, B );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public void decryptBlock( Binary block )
	{
		MUST( block.size() == BLOCK_SIZE, "Incorrect block size" );

		int A = block.getIntLE( 0 );
		int B = block.getIntLE( 4 );

		for( int i = roundsNum; i >= 1; i-- )
		{
			B = rotateRight( B - sKey[ 2 * i + 1 ], A ) ^ A;
			A = rotateRight( A - sKey[ 2 * i ], B ) ^ B;
		}

		block.setIntLE( 0, A - sKey[ 0 ] );
		block.setIntLE( 4, B - sKey[ 1 ] );
	}

}
