// Denom.org
// bouncycastle.org

package org.denom.crypt.streamcipher;

import org.denom.Binary;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

/**
 * Daniel J. Bernstein's XSalsa20 stream cipher - Salsa20 with an extended nonce.
 * Key size = 32 bytes.
 * IV size = 24 bytes.
 */
public class XSalsa20 extends Salsa20
{
	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * 20 rounds.
	 * Set key later.
	 */
	public XSalsa20()
	{
		super( 20, Bin(32) );
	}
	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param rounds the number of rounds (must be an even number).
	 * @param key [16 or 32 bytes]
	 */
	public XSalsa20( final Binary key )
	{
		super( 20, key );
	}

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	public String getAlgName()
	{
		return "XSalsa20";
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param key [32 bytes]
	 */
	@Override
	public void setKey( final Binary key )
	{
		MUST( key.size() == 32, "Wrong key size" );
		this.key = key.clone();
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param IV - 24 bytes.
	 */
	@Override
	public XSalsa20 startEncrypt( final Binary iv )
	{
		MUST( (iv != null) && (iv.size() == 24), "Wrong IV size" );
		initXSalsa( iv );
		reset();
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param IV - 24 bytes.
	 */
	@Override
	public XSalsa20 startDecrypt( final Binary iv )
	{
		MUST( (iv != null) && (iv.size() == 24), "Wrong IV size" );
		initXSalsa( iv );
		reset();
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * XSalsa20 key generation: process 256 bit input key and 128 bits of the input nonce using a
	 * core Salsa20 function without input addition to produce 256 bit working key and use that with
	 * the remaining 64 bits of nonce to initialize a standard Salsa20 engine state.
	 */
	private void initXSalsa( final Binary iv )
	{
		super.initSalsa( iv );

		// Pack next 64 bits of IV into engine state instead of counter
		littleEndianToInt( iv.getDataRef(), 8, engineState, 8, 2 );

		// Process engine state to generate Salsa20 key
		int[] hsalsa20Out = new int[ engineState.length ];
		salsaCore( 20, engineState, hsalsa20Out );

		// Set new key, removing addition in last round of salsaCore
		engineState[ 1 ] = hsalsa20Out[ 0 ] - engineState[ 0 ];
		engineState[ 2 ] = hsalsa20Out[ 5 ] - engineState[ 5 ];
		engineState[ 3 ] = hsalsa20Out[ 10 ] - engineState[ 10 ];
		engineState[ 4 ] = hsalsa20Out[ 15 ] - engineState[ 15 ];

		engineState[ 11 ] = hsalsa20Out[ 6 ] - engineState[ 6 ];
		engineState[ 12 ] = hsalsa20Out[ 7 ] - engineState[ 7 ];
		engineState[ 13 ] = hsalsa20Out[ 8 ] - engineState[ 8 ];
		engineState[ 14 ] = hsalsa20Out[ 9 ] - engineState[ 9 ];

		// Last 64 bits of input IV
		littleEndianToInt( iv.getDataRef(), 16, engineState, 6, 2 );
	}
}
