// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard;

import org.denom.Binary;
import org.denom.format.JSONObject;

/**
 * Классы, реализующие данный интерфейс, обеспечивают взаимодействие с сущностью (устройство, сервис, файл),
 * которая хранит ключи.
 * 
 * SecurityModule вычисляет производные ключи и криптограммы для персонализации и доступа к карточным приложениям;
 * проверяет криптограммы, сгенерированные карточным приложением.
 */
public interface ISecurityModuleBase extends AutoCloseable
{
	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Инициализировать объект параметрами из JSON.
	 * @param jo - параметры SecurityModule.
	 * Подключение к SecurityModule-ю не производится.
	 * @return ссылка на себя.
	 */
	public ISecurityModuleBase init( JSONObject jo );

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Подключиться/установить связь с SecurityModule.
	 */
	public void connect();

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Завершить работу с SecurityModule.
	 */
	public void disconnect();

	// -----------------------------------------------------------------------------------------------------------------
	@Override
	default public void close()
	{
		disconnect();
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Сгенерировать диверсифицированный мастер-ключ (MK) карты из ключа эмитента.
	 * Алгоритм диверсификации: Diversified key = '3DES-ECB encrypt no pad' ( ASN || 'дополнение ASN' ). 
	 * @param keyId - идентификатор ключа (см. EmvKeyID, Smu3Sam.DomainKeyID, Smu3.KeyID).
	 * @param asn - серийный номер приложения [8 байт].
	 * @return - производный ключ [16 байт].
	 */
	public Binary generateMK( int keyId, Binary asn );

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Возвращает размер ключа RSA (модуля N) в байтах.
	 */
	public int rsaGetNLen();

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Выполнить криптографическое преобразование RSA на закрытом ключе.
	 */
	public Binary rsaCryptPrivate( final Binary data );

}
