// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard.cap;

import org.denom.Binary;
import org.denom.Ex;

import static org.denom.Ex.MUST;

/**
 * CAP-file component.
 */
public abstract class CapComponent
{
	// -----------------------------------------------------------------------------------------------------------------
	public final CapFile cap;
	public final String fullName;
	public final int tag;

	// -----------------------------------------------------------------------------------------------------------------
	public CapComponent( CapFile cap, int tag, String fullName )
	{
		this.cap = cap;
		this.tag = tag;
		this.fullName = fullName;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Сериализовать компонент в байтовый массив.
	 */
	public abstract Binary toBin();


	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Теги, идентифицирующие компоненты CAP-файла.
	 */
	public static final int TAG_HEADER        = 1;
	public static final int TAG_DIRECTORY     = 2;
	public static final int TAG_APPLET        = 3;
	public static final int TAG_IMPORT        = 4;
	public static final int TAG_CONSTANT_POOL = 5;
	public static final int TAG_CLASS         = 6;
	public static final int TAG_METHOD        = 7;
	public static final int TAG_STATIC_FIELD  = 8;
	public static final int TAG_REFLOCATION   = 9;
	public static final int TAG_EXPORT        = 10;
	public static final int TAG_DESCRIPTOR    = 11;
	public static final int TAG_DEBUG         = 12;

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Имена компонентов.
	 * Индекс - тег компонента, см. TAG_*.
	 */
	public static final String[] NAMES =
	{
		"",
		"Header.cap",
		"Directory.cap",
		"Applet.cap",
		"Import.cap",
		"ConstantPool.cap",
		"Class.cap",
		"Method.cap",
		"StaticField.cap",
		"RefLocation.cap",
		"Export.cap",
		"Descriptor.cap",
		"Debug.cap"
	};

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Рекомендуемый порядок загрузки компонентов
	 */
	public static final int[] LOAD_ORDER = 
	{
		TAG_HEADER,
		TAG_DIRECTORY,
		TAG_IMPORT,
		TAG_APPLET,
		TAG_CLASS,
		TAG_METHOD,
		TAG_STATIC_FIELD,
		TAG_EXPORT,
		TAG_CONSTANT_POOL,
		TAG_REFLOCATION,
		TAG_DESCRIPTOR // Optional
	};

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @name - Имя компонента
	 * @return тег компонента, либо -1, если имя не совпадает ни с одним компонентом
	 */
	public static int findTag( String name )
	{
		for( int i = 1; i < NAMES.length; ++i )
			if( name.endsWith( NAMES[ i ] ) )
				return i;

		return -1;
	}

	// -----------------------------------------------------------------------------------------------------------------
	public static CapComponent parse( CapFile cap, String name, final Binary fullBody )
	{
		MUST( fullBody.size() > 3, "Wrong CAP component size" );

		int tag = fullBody.get( 0 );
		switch( tag )
		{
			case TAG_HEADER:        return new ComponentHeader      ( cap, name, fullBody );
			case TAG_DIRECTORY:     return new ComponentDirectory   ( cap, name, fullBody );
			case TAG_APPLET:        return new ComponentApplet      ( cap, name, fullBody );
			case TAG_IMPORT:        return new ComponentImport      ( cap, name, fullBody );
			case TAG_CONSTANT_POOL: return new ComponentConstantPool( cap, name, fullBody );
			case TAG_CLASS:         return new ComponentClass       ( cap, name, fullBody );
			case TAG_METHOD:        return new ComponentMethod      ( cap, name, fullBody );
			case TAG_STATIC_FIELD:  return new ComponentStaticField ( cap, name, fullBody );
			case TAG_REFLOCATION:   return new ComponentRefLocation ( cap, name, fullBody );
			case TAG_EXPORT:        return new ComponentExport      ( cap, name, fullBody );
			case TAG_DESCRIPTOR:    return new ComponentDescriptor  ( cap, name, fullBody );
			case TAG_DEBUG:         return new ComponentDebug       ( cap, name, fullBody );
			default:
				throw new Ex( "Wrong CAP component" );
		}
	}

}
