// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard.cap;

import org.denom.Binary;
import org.denom.Ex;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

/**
 * CAP-file component - ConstantPool.
 */
public class ComponentConstantPool extends CapComponent
{
	public int tag;
	public int size;

	public CPInfo[] constPool;

	// -----------------------------------------------------------------------------------------------------------------
	public ComponentConstantPool( CapFile cap, String fullName, final Binary b )
	{
		super( cap, TAG_CONSTANT_POOL, fullName );

		int offs = 0;

		tag = b.get( offs++ );
		size = b.getU16( offs ); offs += 2;

		int count = b.getU16( offs ); offs += 2;
		constPool = new CPInfo[ count ];
		for( int i = 0; i < constPool.length; ++i )
		{
			constPool[ i ] = new CPInfo();
			offs = constPool[ i ].parse( b, offs );
		}

		MUST( (tag == TAG_CONSTANT_POOL) && (size == (b.size() - 3))
				&& (offs == b.size()), "Wrong component ConstantPool" );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public Binary toBin()
	{
		Binary b = Bin().reserve( size + 3 );

		b.add( tag );
		b.addU16( size );

		b.addU16( constPool.length );
		for( CPInfo cp : constPool )
			cp.toBin( b );

		return b;
	}

	// =================================================================================================================
	public static final byte CONSTANT_ClassRef         = 1;
	public static final byte CONSTANT_InstanceFieldRef = 2;
	public static final byte CONSTANT_VirtualMethodRef = 3;
	public static final byte CONSTANT_SuperMethodRef   = 4;
	public static final byte CONSTANT_StaticFieldRef   = 5;
	public static final byte CONSTANT_StaticMethodRef  = 6;

	// =================================================================================================================
	public static class CPInfo implements IComponentElement
	{
		public int tag;
		public IComponentElement info;

		public int parse( Binary b, int offs )
		{
			tag = b.get( offs++ );

			switch( tag )
			{
				case CONSTANT_ClassRef         : info = new ClassRefInfo(); break;
				case CONSTANT_InstanceFieldRef : info = new InstanceRef();  break;
				case CONSTANT_VirtualMethodRef : info = new InstanceRef();  break;
				case CONSTANT_SuperMethodRef   : info = new InstanceRef();  break;
				case CONSTANT_StaticFieldRef   : info = new StaticRef();    break;
				case CONSTANT_StaticMethodRef  : info = new StaticRef();    break;
				default:
					throw new Ex( "Wrong component ConstantPool" );
			}

			offs = info.parse( b, offs );
			return offs;
		}

		public void toBin( Binary b )
		{
			b.add( tag );
			info.toBin( b );
		}
	}

	// =================================================================================================================
	public static class ClassRefInfo implements IComponentElement
	{
		public ClassRef classRef;
		
		public int parse( Binary b, int offs )
		{
			classRef = new ClassRef();
			offs = classRef.parse( b, offs );
			int padding = b.get( offs++ );
			MUST( padding == 0, "Wrong component ConstantPool" );

			return offs;
		}

		public void toBin( Binary b )
		{
			classRef.toBin( b );
			b.add( 0 ); // padding
		}
	}

	// =================================================================================================================
	// Двух-байтовое поле - ссылка на класс внутри пакета или на класс в одном из импортированных пакетов
	public static class ClassRef implements IComponentElement
	{
		public boolean isExternal;

		// Internal
		public int internalClassRef;

		// External
		public int packageToken;
		public int classToken;

		public int parse( Binary b, int offs )
		{
			int b2 = b.getU16( offs );
			offs += 2;

			if( (b2 & 0x8000) != 0 )
			{
				isExternal = true;
				packageToken = (b2 >> 8) & 0x7f;
				classToken = b2 & 0xFF;
			}
			else
			{
				isExternal = false;
				internalClassRef = b2;
			}

			return offs;
		}

		public void toBin( Binary b )
		{
			if( isExternal )
				b.addU16( 0x8000 | (packageToken << 8) | classToken );
			else
				b.addU16( internalClassRef );
		}
	}

	// =================================================================================================================
	// InstanceFieldRef  |  VirtualMethodRef  |  SuperMethodRef
	public static class InstanceRef implements IComponentElement
	{
		public ClassRef classRef;
		public int token;

		public int parse( Binary b, int offs )
		{
			classRef = new ClassRef();
			offs = classRef.parse( b, offs );
			token = b.get( offs++ );
			return offs;
		}

		public void toBin( Binary b )
		{
			classRef.toBin( b );
			b.add( token );
		}
	}

	// =================================================================================================================
	// StaticFieldRef  |  StaticMethodRef
	public static class StaticRef implements IComponentElement
	{
		public boolean isExternal;

		// Internal
		public int offset;

		// External
		public int packageToken;
		public int classToken;
		public int token;

		public int parse( Binary b, int offs )
		{
			int b0 = b.get( offs++ );

			if( (b0 & 0x80) != 0 )
			{
				isExternal = true;
				packageToken = b0 & 0x7F;
				classToken = b.get( offs++ );
				token = b.get( offs++ );
			}
			else
			{
				isExternal = false;
				offset = b.getU16( offs ); offs += 2;
			}

			return offs;
		}

		public void toBin( Binary b )
		{
			if( isExternal )
			{
				b.add( 0x80 | packageToken );
				b.add( classToken );
				b.add( token );
			}
			else
			{
				b.add( 0 ); // padding
				b.addU16( offset );
			}
		}
	}

}
