// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard.cap;

import java.util.ArrayList;

import org.denom.Binary;
import org.denom.smartcard.cap.ComponentDescriptor.MethodDescriptorInfo;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

/**
 * CAP-file component - Method.
 */
public class ComponentMethod extends CapComponent
{
	public int tag;
	public int size;

	public ExceptionHandlerInfo[] exceptionHandlers;
	public ArrayList< MethodInfo > methods;

	// -----------------------------------------------------------------------------------------------------------------
	public ComponentMethod( CapFile cap, String fullName, final Binary b )
	{
		super( cap, TAG_METHOD, fullName );

		int offs = 0;

		tag = b.get( offs++ );
		size = b.getU16( offs ); offs += 2;

		int handlerCount = b.get( offs++ );
		exceptionHandlers = new ExceptionHandlerInfo[ handlerCount ];
		for( int i = 0; i < exceptionHandlers.length; ++i )
		{
			exceptionHandlers[ i ] = new ExceptionHandlerInfo();
			offs = exceptionHandlers[ i ].parse( b, offs );
		}

		methods = new ArrayList<MethodInfo>();
		while( offs < b.size() )
		{
			MethodInfo method = new MethodInfo();
			offs = method.parse( b, offs );
			methods.add( method );
		}

		MUST( (tag == TAG_METHOD) && (size == (b.size() - 3))
				&& (offs == b.size()), "Wrong component Method" );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public Binary toBin()
	{
		Binary b = Bin().reserve( size + 3 );

		b.add( tag );
		b.addU16( size );

		b.add( exceptionHandlers.length );
		for( ExceptionHandlerInfo ex : exceptionHandlers )
			ex.toBin( b );

		for( MethodInfo method : methods )
			method.toBin( b );

		return b;
	}

	// =================================================================================================================
	public static class ExceptionHandlerInfo
	{
		int startOffset;
		int activeLength; // leftmost bit - stop_bit
		int handlerOffset;
		int catchTypeIndex;

		protected int parse( Binary b, int offs )
		{
			startOffset = b.getU16( offs ); offs += 2;
			activeLength = b.getU16( offs ); offs += 2;
			handlerOffset = b.getU16( offs ); offs += 2;
			catchTypeIndex = b.getU16( offs ); offs += 2;
			return offs;
		}

		protected void toBin( Binary b )
		{
			b.addU16( startOffset );
			b.addU16( activeLength );
			b.addU16( handlerOffset );
			b.addU16( catchTypeIndex );
		}
	}

	// =================================================================================================================
	public class MethodInfo
	{
		public static final byte ACC_EXTENDED = 0x8;
		public static final byte ACC_ABSTRACT = 0x4;

		// Method header
		public int flags;
		public int maxStack;
		public int nargs;
		public int maxLocals;

		public MethodDescriptorInfo methodDescr;
		public Binary bytecodes;
		public int bytecodesOffset;
		
		protected int parse( Binary b, int offs )
		{
			int methodOffset = offs - 3;

			// Method header
			int b0 = b.get( offs++ );
			flags = (b0 >> 4) & 0x0f;

			if( (flags & ACC_EXTENDED) != 0 )
			{
				maxStack = b.get( offs++ );
				nargs = b.get( offs++ );
				maxLocals = b.get( offs++ );
			}
			else
			{
				maxStack = b0 & 0x0f;
				int b1 = b.get( offs++ );
				nargs = (b1 >> 4) & 0x0f;
				maxLocals = b1 & 0x0f;
			}

			methodDescr = ((ComponentDescriptor)cap.getComp( TAG_DESCRIPTOR )).findMethod( methodOffset );
			MUST( methodDescr != null, "Wrong component Descriptor or Method. Method info not found" );

			bytecodesOffset = offs - 3;

			bytecodes = b.slice( offs, methodDescr.bytecodeCount );
			offs += methodDescr.bytecodeCount;

			return offs;
		}

		protected void toBin( Binary b )
		{
			if( (flags & ACC_EXTENDED) != 0 )
			{
				b.add( flags << 4 );
				b.add( maxStack );
				b.add( nargs );
				b.add( maxLocals );
			}
			else
			{
				b.add( (flags << 4) | maxStack );
				b.add( (nargs << 4) | maxLocals );
			}

			b.add( bytecodes );
		}
	}
}
