// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard.cap;

import org.denom.Ex;

import org.denom.Binary;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;

/**
 * Парсинг EXP-файла.
 */
public class ExpFile
{
	public int magic;
	public int minorVersion;
	public int majorVersion;

	public ICPInfo[] constPool;
	public int thisPackage;

	public ClassInfo[] classes;

	// -----------------------------------------------------------------------------------------------------------------
	public ExpFile() {}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Считать exp-файл.
	 * @param expPath путь к файлу.
	 * @return this;
	 */
	public ExpFile load( String expPath )
	{
		Binary b = Bin().loadFromFile( expPath );
		parse( b );
		return this;
	}

	// =================================================================================================================
	public static final byte CONSTANT_Package  = 13;
	public static final byte CONSTANT_ClassRef = 7;
	public static final byte CONSTANT_Integer  = 3;
	public static final byte CONSTANT_Utf8     = 1;

	// =================================================================================================================
	public static interface ICPInfo
	{
		int parse( Binary b, int offs );
		
		int getTag();
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Распарсить тело exp-файла.
	 * @return this;
	 */
	public ExpFile parse( Binary b )
	{
		int offs = 0;

		magic = b.getIntBE( offs ); offs += 4;
		minorVersion = b.get( offs++ );
		majorVersion = b.get( offs++ );

		int constPoolCount = b.getU16( offs ); offs += 2;
		constPool = new ICPInfo[ constPoolCount ];

		for( int i = 0; i < constPoolCount; ++i )
		{
			int tag = b.get( offs++ );

			switch( tag )
			{
				case CONSTANT_Package:  constPool[ i ] = new CP_PackageInfo();  break;
				case CONSTANT_ClassRef: constPool[ i ] = new CP_ClassRefInfo(); break;
				case CONSTANT_Integer:  constPool[ i ] = new CP_IntegerInfo();  break;
				case CONSTANT_Utf8:     constPool[ i ] = new CP_Utf8Info();     break;
				default:
					throw new Ex( "Wrong EXP-file" );
			}

			offs = constPool[ i ].parse( b, offs );
		}

		thisPackage = b.getU16( offs ); offs += 2;

		int expClassCount = b.get( offs++ );
		classes = new ClassInfo[ expClassCount ];
		for( int i = 0; i < expClassCount; ++i )
		{
			classes[ i ] = new ClassInfo();
			offs = classes[ i ].parse( b, offs );
		}

		MUST( (magic == 0x00FACADE) && (offs == b.size()), "Wrong EXP-file" );
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Найти информацию о классе по его имени.
	 * @return null - если не найден класс с заданным именем.
	 */
	public ClassInfo findClass( String className )
	{
		for( ClassInfo classInfo : classes )
		{
			ICPInfo cpInfo = constPool[ classInfo.nameIndex ];
			MUST( cpInfo.getTag() == CONSTANT_ClassRef, "Wrong EXP-file" );

			int cpIndex = ((CP_ClassRefInfo)cpInfo).nameIndex;
			cpInfo = constPool[ cpIndex ];
			MUST( cpInfo.getTag() == CONSTANT_Utf8, "Wrong EXP-file" );
			
			String name = ((CP_Utf8Info)cpInfo).val;
			if( name.equals( className ) )
				return classInfo;
		}
		return null;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Найти информацию о методе в заданном классе.
	 * @return null - если не найден класс или метод в нём.
	 */
	public MethodInfo findMethod( String className, String methodName, String methodDescriptor )
	{
		ClassInfo cl = findClass( className );
		if( cl == null )
			return null;

		return cl.findMethod( methodName, methodDescriptor );
	}

	// =================================================================================================================
	public static class CP_PackageInfo implements ICPInfo
	{
		public final static int ACC_LIBRARY = 0x01;

		public int flags;
		public int nameIndex;
		public int minorVersion;
		public int majorVersion;
		public Binary aid;

		public int parse( Binary b, int offs )
		{
			flags = b.get( offs++ );
			nameIndex = b.getU16( offs ); offs += 2;
			minorVersion = b.get( offs++ );
			majorVersion = b.get( offs++ );
			int aidLen = b.get( offs++ );
			aid = b.slice( offs, aidLen );
			offs += aidLen;

			return offs;
		}

		public int getTag()
		{
			return CONSTANT_Package;
		}
	}

	// =================================================================================================================
	public static class CP_ClassRefInfo implements ICPInfo
	{
		public int nameIndex;

		public int parse( Binary b, int offs )
		{
			nameIndex = b.getU16( offs ); offs += 2;
			return offs;
		}

		public int getTag()
		{
			return CONSTANT_ClassRef;
		}
	}

	// =================================================================================================================
	public static class CP_IntegerInfo implements ICPInfo
	{
		public int val;

		public int parse( Binary b, int offs )
		{
			val = b.getIntBE( offs ); offs += 4;
			return offs;
		}

		public int getTag()
		{
			return CONSTANT_Integer;
		}
	}

	// =================================================================================================================
	public static class CP_Utf8Info implements ICPInfo
	{
		public String val;

		public int parse( Binary b, int offs )
		{
			int len = b.getU16( offs ); offs += 2;
			val = b.slice( offs, len ).asUTF8();
			offs += len;
			return offs;
		}

		public int getTag()
		{
			return CONSTANT_Utf8;
		}
	}

	// =================================================================================================================
	public final static int ACC_PUBLIC    = 0x0001;
	public final static int ACC_PROTECTED = 0x0004;
	public final static int ACC_STATIC    = 0x0008;
	public final static int ACC_FINAL     = 0x0010;
	public final static int ACC_INTERFACE = 0x0200;
	public final static int ACC_ABSTRACT  = 0x0400;
	public final static int ACC_SHAREABLE = 0x0800;
	public final static int ACC_REMOTE    = 0x1000;

	// =================================================================================================================
	public class ClassInfo
	{
		public int token;
		public int accessFlags;
		public int nameIndex;
		public int[] supers;
		public int[] interfaces;
		public FieldInfo[] fields;
		public MethodInfo[] methods;


		public int parse( Binary b, int offs )
		{
			token = b.get( offs++ );
			accessFlags = b.getU16( offs ); offs += 2;
			nameIndex = b.getU16( offs ); offs += 2;

			int count = b.getU16( offs ); offs += 2;
			supers = new int[ count ];
			for( int i = 0; i < supers.length; ++i )
			{
				supers[ i ] = b.getU16( offs ); offs += 2;
			}

			count = b.get( offs++ );
			interfaces = new int[ count ];
			for( int i = 0; i < interfaces.length; ++i )
			{
				interfaces[ i ] = b.getU16( offs ); offs += 2;
			}

			count = b.getU16( offs ); offs += 2;
			fields = new FieldInfo[ count ];
			for( int i = 0; i < fields.length; ++i )
			{
				fields[ i ] = new FieldInfo();
				offs = fields[ i ].parse( b, offs );
			}

			count = b.getU16( offs ); offs += 2;
			methods = new MethodInfo[ count ];
			for( int i = 0; i < methods.length; ++i )
			{
				methods[ i ] = new MethodInfo();
				offs = methods[ i ].parse( b, offs );
			}

			return offs;
		}

		// -----------------------------------------------------------------------------------------------------------------
		public MethodInfo findMethod( String name, String descriptor )
		{
			for( MethodInfo method : methods )
			{
				ICPInfo cpInfo = constPool[ method.nameIndex ];
				MUST( cpInfo.getTag() == CONSTANT_Utf8, "Wrong EXP-file" );
				String curMethodName = ((CP_Utf8Info)cpInfo).val;

				if( !name.equals( curMethodName ) )
					continue;

				cpInfo = constPool[ method.descriptorIndex ];
				MUST( cpInfo.getTag() == CONSTANT_Utf8, "Wrong EXP-file" );
				String curDescriptor = ((CP_Utf8Info)cpInfo).val;

				if( descriptor.equals( curDescriptor ) )
					return method;
			}

			return null;
		}
	}

	// =================================================================================================================
	public static class AttributeInfo
	{
		int nameIndex;
		Binary info;

		public int parse( Binary b, int offs )
		{
			nameIndex = b.getU16( offs ); offs += 2;
			int len = b.getIntBE( offs ); offs += 4;
			info = b.slice( offs, len );
			offs += len;
			return offs;
		}
	}

	// =================================================================================================================
	public static class FieldInfo
	{
		public int token;
		public int accessFlags;
		public int nameIndex;
		public int descriptorIndex;
		public AttributeInfo[] attributes;

		public int parse( Binary b, int offs )
		{
			token = b.get( offs++ );
			accessFlags = b.getU16( offs ); offs += 2;
			nameIndex = b.getU16( offs ); offs += 2;
			descriptorIndex = b.getU16( offs ); offs += 2;

			int count = b.getU16( offs ); offs += 2;
			attributes = new AttributeInfo[ count ];
			
			for( int i = 0; i < attributes.length; ++i )
			{
				attributes[ i ] = new AttributeInfo();
				offs = attributes[ i ].parse( b, offs );
			}

			return offs;
		}
	}

	// =================================================================================================================
	public static class MethodInfo
	{
		public int token;
		public int accessFlags;
		public int nameIndex;
		public int descriptorIndex;

		public int parse( Binary b, int offs )
		{
			token = b.get( offs++ );
			accessFlags = b.getU16( offs ); offs += 2;
			nameIndex = b.getU16( offs ); offs += 2;
			descriptorIndex = b.getU16( offs ); offs += 2;

			return offs;
		}
	}
}
