// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard.emv.kernel8;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Map;

import org.denom.*;
import org.denom.crypt.blockcipher.*;
import org.denom.crypt.ec.ECAlg;
import org.denom.crypt.ec.Fp.FpCurveAbstract;
import org.denom.format.BerTLV;
import org.denom.format.BerTLVList;
import org.denom.smartcard.emv.TagEmv;
import org.denom.smartcard.emv.kernel8.struct.TagKernel8;
import org.denom.smartcard.emv.kernel8.struct.TlvDatabase;

import static org.denom.Binary.Bin;
import static org.denom.Ex.MUST;
import static org.denom.format.BerTLV.Tlv;

/**
 * Вычисление криптограмм по EMV-спецификации для Kernel 8.
 * EMV Contactless Specifications for Payment Systems, Book E, Security and Key Management.
 * Все функции статические.
 */
public final class Kernel8Crypt
{
	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Сгенерировать EC ключ для карты или эмитента.
	 * Book E, 8.8.5
	 * @param ecAlg с кривой FP
	 * @return Только X-координата публичного ключа.
	 */
	public static Binary generateEccKey( ECAlg ecAlg )
	{
		FpCurveAbstract curve = (FpCurveAbstract)ecAlg.getCurve();
		int nSize = curve.getNField();

		BigInteger p = curve.getP();
		BigInteger p2 = p.add( BigInteger.ONE ).divide( BigInteger.valueOf( 2 ) );

		// Repeatedly generate until the y-coordinate of d*G is less than (p+1)/2
		BigInteger y = null;
		Binary pub;
		do
		{
			ecAlg.generateKeyPair();
			pub = ecAlg.getPublic( false );
			y = new BigInteger( Bin( Bin( 1 ), pub.last( nSize ) ).getBytes() );
		}
		while( y.compareTo( p2 ) != -1 );

		return pub.slice( 1, nSize );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Book E, 8.8.4 Point Finding.
	 * @param publicKeyX - Координата X.
	 */
	public static void restorePublicKey( ECAlg ecAlg, final Binary publicKeyX )
	{
		int NField = ((FpCurveAbstract)ecAlg.getCurve()).getNField();

		ecAlg.setPublic( Bin(1, 0x02).add( publicKeyX ) );
		Binary key02 = ecAlg.getPublic( false );
		key02 = key02.last( NField );

		ecAlg.setPublic( Bin(1, 0x03).add( publicKeyX ) );
		Binary key03 = ecAlg.getPublic( false );
		key03 = key03.last( NField );

		// из двух точек берем ту, у которой Y меньше.
		if( key02.compareTo( key03 ) == -1 )
			ecAlg.setPublic( Bin(1, 0x02).add( publicKeyX ) );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычислить Card blinded public key - Pc.
	 * Карта вычисляет Pc и передаёт терминалу в GPO Response.
	 * Book E, 7.1  BDH Initialisation.
	 * @param r - blinding factor, длина = размеру параметров кривой. 1 < r < n-1.
	 * @param Qc - публичный ключ карты.
	 * @return Pc = r * Qc.
	 */
	public static Binary BDHCalcPc( final ECAlg Qc, final Binary r )
	{
		ECAlg ecAlg = Qc.clone();
		ecAlg.setPrivate( r );
		Binary Pc = ecAlg.calcDH( Qc.getPublic( false ) );
		return Pc;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычислить BDH shared secret (z) со стороны карты.
	 * Карта вычисляет его в GPO для генерации сессионных ключей.
	 * Book E, 7.2  BDH Key Derivation.
	 * @param Qk - Публичный ECC ключ терминала.
	 * @param Dc - секретный ключ карты.
	 * @param r - blinding factor, длина = размеру параметров кривой. 1 < r < n-1.
	 * @return z - shared secret = Dc * r * Qk.
	 */
	public static Binary BDHCalcZ( final ECAlg Qk, final Binary Dc, final Binary r )
	{
		ECAlg ecAlg = Qk.clone();
		ecAlg.setPrivate( Dc );
		Binary p = Bin(1, 0x02).add( ecAlg.calcDH( Qk.getPublic( false ) ) );
		ecAlg.setPrivate( r );
		Binary z = ecAlg.calcDH( p );
		return z;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычислить BDH shared secret (z) со стороны терминала.
	 * Book E, 7.2  BDH Key Derivation.
	 * @param Dk - Секретный ключ терминала.
	 * @param Pc - Card blinded public key, длина = размеру параметров кривой. То, что карта возвращает в GPO.
	 * @return z - shared secret = Dk * Pc.
	 */
	public static Binary BDHCalcZ( final ECAlg Dk, final Binary Pc )
	{
		Binary PcPoint = Bin(1, 0x02).add( Pc );
		Binary z = Dk.calcDH( PcPoint );
		return z;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычислить 16-байтный производный ключ - Kd.
	 * Book E, 7.2  BDH Key Derivation.
	 * @param z - shared secret, см. BDHCalcSecret.
	 * @return Kd - derivation key [16 байт].
	 */
	public static Binary BDHCalcKd( final Binary z )
	{
		// Let V be 16 zero bytes.
		// Kd = AES-CMAC (V) [Z]
		AES aes = new AES( Bin(16) );
		Binary Kd = aes.calcCMAC( z, null );
		return Kd;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param b1 первый байт в данных для вывода ключа - "номер ключа" - 01, 02, 03.
	 * SK = AES (Kd) [ b1  || '01' || '00' || '54334A325957773D' || 'A5A5A5'|| '0180']
	 * @return sessionKey [16 байт]
	 */
	public static Binary BDHCalcSessionKey( final Binary Kd, int b1 )
	{
		AES aes = new AES( Kd );
		Binary b = Bin( 1, b1 ).add( "01 00 54334A325957773D A5A5A5 0180" );
		aes.encryptBlock( b );
		return b;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Derive the session key for confidentiality SKc.
	 * Book E, 7.2  BDH Key Derivation.
	 * SKc = AES (Kd) ['01' || '01' || '00' || '54334A325957773D' || 'A5A5A5'|| '0180']
	 * @return SKc [16 байт]
	 */
	public static Binary BDHCalcSKc( final Binary Kd )
	{
		return BDHCalcSessionKey( Kd, 0x01 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Derive the session key for integrity SKi.
	 * Book E, 7.2  BDH Key Derivation.
	 * SKi = AES (Kd) ['02' || '01' || '00' || '54334A325957773D' || 'A5A5A5'|| '0180']
	 * @return SKi [16 байт]
	 */
	public static Binary BDHCalcSKi( final Binary Kd )
	{
		return BDHCalcSessionKey( Kd, 0x02 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычислить сессионные ключи BDH.
	 * @param Dk - Секретный ключ терминала.
	 * @param Pc - Card blinded public key, длина = размеру параметров кривой. То, что карта возвращает в GPO.
	 * @return { SKc, SKi }
	 */
	public static Binary[] BDHCalcSessionKeys( final ECAlg Dk, final Binary Pc )
	{
		Binary z = BDHCalcZ( Dk, Pc );
		Binary Kd = BDHCalcKd( z );
		Binary SKc = BDHCalcSKc( Kd );
		Binary SKi = BDHCalcSKi( Kd );
		return new Binary[] { SKc, SKi };
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * 7.3  BDH Blinding Factor Validation.
	 * @param Pc - blinded card public key.
	 */
	public static boolean BDHValidate( ECAlg Qc, final Binary r, final Binary Pc )
	{
		ECAlg ec = Qc.clone();
		ec.setPrivate( r );
		Binary QcPoint = Qc.getPublic( false );
		Binary myPc = ec.calcDH( QcPoint );
		return Pc.equals( myPc );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Book E,  8.6.2  CTR.
	 * @param messageCounter (U16).
	 * @param data input message (plain or crypt).
	 * @return output message C (crypt or plain).
	 */
	public static Binary cryptCTR( BlockCipher SessionKey, int messageCounter, final Binary data )
	{
		MUST( Int.isU16( messageCounter ), "Wrong messageCounter for crypt CTR" );
		Binary SV = Bin( SessionKey.getBlockSize() );
		SV.setU16( 0, messageCounter );
		return SessionKey.cryptCTR( data, SV );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param Er - encrypted blinding factor.
	 * @param CMC - Card Message Counter.
	 * Recover 'r' from 'Encrypted r'
	 */
	public static Binary BDHRecoverR( BlockCipher SKc, final Binary Er, int CMC )
	{
		return cryptCTR( SKc, CMC, Er );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Конкатенация данных для вычисления SDA Hash.
	 * @param sdaRecords - Все записи, участвующие в SDA.
	 * @param extendedSDARelData - конкатенация всех TLV из списка 'Extended SDA Tag List'.
	 * @param aip [2 байта] без тега.
	 */
	public static Binary formSDAData( final Binary sdaRecords, Binary extendedSDARelData, final Binary aip )
	{
		return Bin().add( sdaRecords ).add( extendedSDARelData ).add( aip );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Сконкатенировать записи, участвующие в SDA по Kernel 8.
	 * Отбрасывается тег 70 для всех записей.
	 */
	public static Binary getSdaRecords( Map<Binary, Binary> records, Arr<Binary> sdaRecIds )
	{
		Binary sdaRecords = Bin();
		for( Binary sdaRecId : sdaRecIds )
		{
			Binary record = records.get( sdaRecId );

			// EMV 4.3, Book 3, 10.3 - Offline Data Authentication.
			// Все записи, участвующие в SDA, должны быть в теге 0x70.
			BerTLV rec = new BerTLV( record );
			MUST( rec.tag == TagEmv.ReadRecordResponseMessageTemplate, "Record body for SDA must be in tag 0x70" );

			// тег 0x70 отбрасывается, берётся только Value.
			sdaRecords.add( rec.value );
		}
		return sdaRecords;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @return Конкатенация TLV-объектов согласно списка 'Extended SDA Tag List'.
	 */
	public static Binary createExtSDARelData( TlvDatabase tlvDB )
	{
		Binary res = Bin();

		if( !tlvDB.IsNotEmpty( TagKernel8.ExtendedSDATagList ) )
			return res;

		Binary extTagList = tlvDB.GetValue( TagKernel8.ExtendedSDATagList );
		Arr<Integer> tags = BerTLV.parseTagList( extTagList );
		for( int tag : tags )
		{
			Binary val = tlvDB.GetValue( tag );
			if( val != null )
				res.add( Tlv( tag, val ) );
			else
				res.add( Tlv( tag, Bin() ) );
		}

		return res;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * 8.6.5  AES-CMAC+
	 */
	public static Binary calcAesCmacPlus( AES aes, Binary message )
	{
		Binary cmac = aes.calcCMAC( message, null );
		cmac = aes.decrypt( cmac, CryptoMode.CBC, AlignMode.NONE, cmac );
		return cmac;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Calc IAD-MAC. Book E, 3.2  Local Cryptogram.
	 * IAD-MAC = Leftmost 8 bytes of AES-CMAC+ (SKi) [0000 || Input Data] 
	 */
	public static Binary calcIADMac( AES aesSKi, Binary inputData )
	{
		Binary b = Bin( 2 ).add( inputData );
		Binary cmac = calcAesCmacPlus( aesSKi, b );
		return cmac.first( 8 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Calc IAD-MAC. Book E, 3.2  Local Cryptogram.
	 * Input data:
	 *  • PDOL Values (Value field of PDOL Related Data)
	 *  • CDOL1 Related Data
	 *  • (optional) Terminal Relay Resistance Entropy
	 *  • (optional) Last ERRD Response (without tag '80' and length '0A')
	 *  • GENERATE AC Response = TLV with tag 0x77
	 *  • SDA Hash (hash over the Card Static Data to be Authenticated)
	 *  
	 * @param terminalRREntropy - can be null or empty.
	 * @param lastERRDResponse - can be null or empty
	 */
	public static Binary formDataForIADMac( final Binary PDOLValues, final Binary CDOL1RelatedData,
			final Binary terminalRREntropy, final Binary lastERRDResponse,
			final Binary genAcResponse, final Binary sdaHash )
	{
		Binary data = Bin().reserve( 300 );
		data.add( PDOLValues );
		data.add( CDOL1RelatedData );
		if( (terminalRREntropy != null) && (lastERRDResponse != null) )
		{
			data.add( terminalRREntropy );
			data.add( lastERRDResponse );
		}

		// GENERATE AC Response Message without:
		//   -- Application Cryptogram
		//   -- EDA-MAC
		//   -- Tag '77' and length
		ArrayList<BerTLV> tlvs = new BerTLVList( new BerTLV( genAcResponse ).value ).recs;
		for( BerTLV tlv : tlvs )
			if( (tlv.tag != TagEmv.ApplicationCryptogram) && (tlv.tag != TagKernel8.EDA_MAC) )
				data.add( tlv.toBin() );

		data.add( sdaHash );
		
		return data;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Calc EDA-MAC. Book E, 3.2  Local Cryptogram.
	 */
	public static Binary calcEDAMac( AES SKi, final Binary ac, final Binary iadMac )
	{
		// EDA-MAC = Leftmost 8 bytes of AES-CMAC (SKi) ['0000' || AC || IAD-MAC]
		Binary data = Bin( 2 ).add( ac ).add( iadMac );
		Binary edaMac = SKi.calcCMAC( data, null ).first( 8 );
		return edaMac;
	}

	// =================================================================================================================
	// Data Storage
	// =================================================================================================================

	// -----------------------------------------------------------------------------------------------------------------
	public static Binary calcMac( BlockCipher alg, int messageCounter, final Binary data )
	{
		MUST( Int.isU16( messageCounter ), "Wrong messageCounter for calcMac" );
		Binary b = Bin().addU16( messageCounter ).add( data );
		return alg.calcCMAC( b, null ).first( 8 );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Зашифровать Data Envelope + MAC.
	 * Формирование ответа карты на READ DATA.
	 */
	public static Binary encryptReadData( BlockCipher SKc, BlockCipher SKi, int CMC, final Binary plainTlv )
	{
		Binary encryptedTLV = cryptCTR( SKc, CMC, plainTlv );
		Binary mac = calcMac( SKi, CMC, encryptedTLV );
		return encryptedTLV.add( mac );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Расшифровать ответ карты на READ DATA.
	 * @return Plain TLV или null, если MAC не совпал.
	 */
	public static Binary decryptReadData( BlockCipher SKc, BlockCipher SKi, int CMC, final Binary crypt )
	{
		if( crypt.size() < 8 )
			return null;

		Binary cardMac = crypt.last( 8 );
		Binary encryptedTLV = crypt.first( crypt.size() - 8 );

		// CMC для MAC и Decrypt один и тот же
		Binary myMac = calcMac( SKi, CMC, encryptedTLV );
		if( !myMac.equals( cardMac ) )
			return null;

		Binary plainTLV = cryptCTR( SKc, CMC, encryptedTLV );
		return plainTLV;
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Зашифровать/Расшифровать Data Envelope для отправки в карту в команде WRITE DATA.
	 */
	public static Binary cryptWriteData( BlockCipher SKc, int KMC, final Binary plainTlv )
	{
		return cryptCTR( SKc, KMC, plainTlv );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * Вычислить MAC от Plain TLV  Data Envelope для ответа карты на WRITE DATA
	 */
	public static Binary calcDataEnvelopeMac( BlockCipher SKi, int CMC, final Binary plainTlv )
	{
		return calcMac( SKi, CMC, plainTlv );
	}

}
