// Denom.org
// Author:  Sergey Novochenko,  Digrol@gmail.com

package org.denom.smartcard.emv.kernel8;

import java.util.*;
import org.denom.*;
import org.denom.log.*;
import org.denom.format.*;
import org.denom.smartcard.*;
import org.denom.smartcard.emv.*;
import org.denom.smartcard.emv.kernel8.struct.*;

import static org.denom.Ex.MUST;
import static org.denom.Binary.*;

/**
 * Work with card application by commands from 'EMV Сontactless Book C-8, Kernel 8 Specification v1.1'.
 * It is not Kernel 8 flow.
 * For tests and research.
 */
public class TerminalK8
{
	/**
	 * Card reader to send commands to.
	 */
	public CardReader cr = null;

	public ILog log = new LogDummy();

	/**
	 * AID of application instance.
	 */
	public Binary appAid = Bin();

	public Map<Binary, Binary> caPublicKeys = new LinkedHashMap<>();

	// -----------------------------------------------------------------------------------------------------------------
	public TlvDatabase tlvDB;

	public IKernel8Crypter crypter;

	public Binary aip = Bin();
	public Binary pdolValues = Bin();
	public Binary IadMac;
	public Binary sdaRecords = Bin();
	public Binary lastERRDResponse;

	// -----------------------------------------------------------------------------------------------------------------
	public TerminalK8() {}

	// -----------------------------------------------------------------------------------------------------------------
	public TerminalK8( CardReader cr, final Binary aid, IKernel8Crypter crypter )
	{
		this.cr = cr;
		this.appAid = aid.clone();
		this.crypter = crypter;
	}

	// -----------------------------------------------------------------------------------------------------------------
	public void resetSession()
	{
		tlvDB = new TlvDatabase( new TagDictKernel8() );

		aip.clear();
		pdolValues.clear();

		sdaRecords = Bin();
		lastERRDResponse = null;
		IadMac = Bin();

		crypter.resetSession();
	}

	// -----------------------------------------------------------------------------------------------------------------
	public TerminalK8 setReader( CardReader reader )
	{
		this.cr = reader;
		resetSession();
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	public TerminalK8 setAID( final Binary aid )
	{
		this.appAid = aid.clone();
		resetSession();
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	public TerminalK8 setLog( ILog log )
	{
		this.log = log;
		return this;
	}

	// -----------------------------------------------------------------------------------------------------------------
	public void addCAPublicKey( final Binary caPKIndex, final Binary caPublicKey )
	{
		caPublicKeys.put( caPKIndex, caPublicKey );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public Binary getVal( int tag )
	{
		return tlvDB.GetValue( tag );
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void processFCI( Binary fci )
	{
		MUST( BerTLV.isTLV( fci ), "Not TLV in response on SELECT" );
		BerTLV tlv = new BerTLV( fci );
		MUST( tlv.tag == TagEmv.FCI, "Not FCI on Select" );

		// Save whole FCI
		tlvDB.store( tlv );
		// And primitive tags in it
		MUST( tlvDB.ParseAndStoreCardResponse( fci, false ), "Wrong FCI" );
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * SELECT card application.
	 * @return card response - FCI.
	 */
	public Binary select()
	{
		cr.Cmd( ApduIso.SelectAID( appAid ), RApdu.ST_ANY );
		MUST( cr.rapdu.isOk() || (cr.rapdu.sw1() == 0x62) || (cr.rapdu.sw1() == 0x63),
			"Can't select card application, status: " + Num_Bin( cr.rapdu.status, 2 ).Hex() );

		resetSession();

		processFCI( cr.resp );
		return cr.resp.clone();
	}

	// -----------------------------------------------------------------------------------------------------------------
	public void getProcessingOptions()
	{
		Binary fci = tlvDB.GetTLV( TagEmv.FCI );
		resetSession();

		if( !fci.empty() )
			processFCI( fci );

		tlvDB.store( TagEmv.TerminalVerificationResults, Bin( 5 ) );

		tlvDB.store( TagEmv.UnpredictableNumber, Bin().random( 4 ) );

		Binary cardQualifier = tlvDB.GetValue( TagKernel8.CardQualifier );
		MUST( cardQualifier != null, "CardQualifier absent" );
		MUST( crypter.isASISupported( cardQualifier ), "ASI not supported" );

		tlvDB.store( TagKernel8.KernelKeyData, crypter.getKernelKeyData() );
		tlvDB.store( TagKernel8.KernelQualifier, crypter.getKernelQualifier() );

		// Search PDOL in tlvDB and form PDOL Values (without tag 0x83)
		if( tlvDB.IsNotEmpty( TagEmv.PDOL ) )
			pdolValues = tlvDB.formDOLValues( tlvDB.GetValue( TagEmv.PDOL ) );

		Cmd( ApduEmv.GetProcessingOptions( pdolValues ) );

		BerTLV tlvResp = new BerTLV( cr.resp );
		MUST( tlvResp.tag == TagEmv.ResponseMessageTemplateFormat2, "Wrong TLV in GPO response" );

		MUST( tlvDB.ParseAndStoreCardResponse( cr.resp, false ), "Wrong GPO Response" );

		Binary cardKeyData = tlvDB.GetValue( TagKernel8.CardKeyData );
		MUST( crypter.processCardKeyData( cardKeyData ), "Wrong GPO Response" );
		
		this.aip = tlvDB.GetValue( TagEmv.AIP );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public Binary exchangeRRData()
	{
		Cmd( ApduEmv.ExchangeRelayResistanceData( tlvDB.GetValue( TagEmv.UnpredictableNumber ) ) );

		BerTLV tlv = new BerTLV( cr.resp );
		MUST( (tlv.tag == 0x80) && (tlv.value.size() == 10), "Wrong response on ExchangeRelayResistanceData" );

		lastERRDResponse = tlv.value;

		return tlv.value;
	}

	// -----------------------------------------------------------------------------------------------------------------
	public void readAFLRecords()
	{
		Binary afl = tlvDB.GetValue( TagEmv.AFL );

		Arr<Binary> sdaRecIds = new Arr<Binary>();
		Map<Binary, Binary> records = EmvUtil.readAflRecords( cr, EmvUtil.parseAFL( afl, sdaRecIds ) );

		for( Binary rec : records.values() )
		{
			rec.assign( crypter.decryptRecord( rec ) );
			tlvDB.ParseAndStoreCardResponse( rec, false );
		}

		sdaRecords = Kernel8Crypt.getSdaRecords( records, sdaRecIds );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public Binary getData( int tag )
	{
		cr.Cmd( ApduEmv.GetData( tag ) );
		return cr.resp.clone();
	}

	// -----------------------------------------------------------------------------------------------------------------
	public Binary readLog( int sfi, int recordId )
	{
		Cmd( ApduIso.ReadRecord( sfi, recordId ) );
		return cr.resp.clone();
	}

	// -----------------------------------------------------------------------------------------------------------------
	public Binary readRecord( int sfi, int recordId )
	{
		Cmd( ApduIso.ReadRecord( sfi, recordId ) );
		return crypter.decryptRecord( cr.resp );
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void processCertificates()
	{
		Binary b = tlvDB.GetValue( TagEmv.CAPublicKeyIndexICC );
		MUST( b != null, "CA PK Index absent on card" );

		Binary caPublicKey = caPublicKeys.get( b );
		MUST( caPublicKey != null, "Unknown CA Key" );

		Binary issuerCertBin = tlvDB.GetValue( TagEmv.IssuerPublicKeyCertificate );
		MUST( issuerCertBin != null, "Issuer Public Key Cert absent on card" );

		Binary iccCertBin = tlvDB.GetValue( TagEmv.ICC_PublicKeyCertificate );
		MUST( iccCertBin != null, "ICC Public Key Cert absent on card" );

		Binary aid = tlvDB.GetValue( TagEmv.DFName );
		Binary pan = tlvDB.GetValue( TagEmv.PAN );

		MUST( crypter.processCertificates( b, caPublicKey, issuerCertBin, iccCertBin, aid, pan ), "Local authentication failed" );
	}

	// -----------------------------------------------------------------------------------------------------------------
	private void processIadMac( Binary genACResponse, Binary cdolRelData )
	{
		Binary terminalRREntropy = null;
		if( lastERRDResponse != null )
			terminalRREntropy = tlvDB.GetValue( TagEmv.UnpredictableNumber );

		IadMac = crypter.calcIADMac( pdolValues, cdolRelData, terminalRREntropy, lastERRDResponse, genACResponse );
		tlvDB.store( TagKernel8.IAD_MAC, IadMac );

		Binary iad = tlvDB.GetRef( TagEmv.IAD );
		int copyIadMac = (aip.get( 1 ) & 0b00000110) >> 1;

		if( copyIadMac == 0b10 )
		{
			Binary val = tlvDB.GetValue( TagKernel8.IAD_MACOffset );
			if( !tlvDB.IsNotEmpty( TagKernel8.IAD_MACOffset ) || (val.asU16() + 8) > iad.size() )
				throw new Ex( "Wrong IAD-MAC Offset" );
			iad.set( val.asU16(), IadMac, 0, IadMac.size() );
		}
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @param cryptType = A.1.108  Reference Control Parameter
	 * @return Ответ карты
	 */
	public Binary generateAC( int cryptType, TlvDatabase params, boolean moreCommands )
	{
		tlvDB.append( params );

		Binary cdolRelData = tlvDB.formDOLValues( tlvDB.GetValue( TagEmv.CDOL1 ) );

		crypter.calcSdaHash( Kernel8Crypt.formSDAData( sdaRecords, Kernel8Crypt.createExtSDARelData( tlvDB ), aip ) );
		processCertificates();

		Cmd( ApduEmv.GenerateAC( cryptType, cdolRelData, moreCommands ) );

		BerTLV tlv77 = new BerTLV( cr.resp );
		MUST( tlv77.tag == TagEmv.ResponseMessageTemplateFormat2, "Wrong Tag in GENERATE AC Response" );
		MUST( tlvDB.ParseAndStoreCardResponse( cr.resp, false ), "Wrong GENERATE AC Response" );

		processIadMac( cr.resp, cdolRelData );

		Binary cardAC = tlvDB.GetValue( TagEmv.ApplicationCryptogram );
		MUST( cardAC != null, "Card AC absent" );

		Binary cardEdaMac = tlvDB.GetValue( TagKernel8.EDA_MAC );
		MUST( cardEdaMac != null, "Card EDA-MAC absent" );

		Binary myEdaMac = crypter.calcEDAMac( IadMac, cardAC );
		MUST( myEdaMac.equals( cardEdaMac ), "Wrong card EDA-MAC" );

		if( tlvDB.IsNotEmpty( TagKernel8.CardTVR ) )
		{
			Binary cardTvr = tlvDB.GetValue( TagKernel8.CardTVR );
			tlvDB.store( TagEmv.TerminalVerificationResults, cardTvr );
		}

		return cr.resp.clone();
	}

	// -----------------------------------------------------------------------------------------------------------------
	/**
	 * @return Plain TLV или null, Если MAC не совпал.
	 */
	public Binary ReadData( int tag )
	{
		Cmd( ApduEmv.ReadData( tag ) );
		return crypter.decryptReadData( cr.resp );
	}

	// -----------------------------------------------------------------------------------------------------------------
	public void WriteData( final Binary plainTlv, boolean moreCommands )
	{
		Binary encryptedTlv = crypter.cryptWriteData( plainTlv );
		Cmd( ApduEmv.WriteData( encryptedTlv, moreCommands ) );

		Binary myMac = crypter.calcDataEnvelopeMac( plainTlv );
		MUST( myMac.equals( cr.resp ), "Wrong card MAC" );
	}

	// -----------------------------------------------------------------------------------------------------------------
	private final String thisClassName = TerminalK8.class.getName();

	// -----------------------------------------------------------------------------------------------------------------
	private void Cmd( CApdu capdu )
	{
		cr.callerClassName = thisClassName;
		cr.Cmd( capdu, RApdu.ST_OK );
	}

}
