/*
 * Decompiled with CFR 0.152.
 */
package org.denom.crypt.ec;

import java.math.BigInteger;
import java.security.SecureRandom;
import java.util.Random;
import org.denom.Binary;
import org.denom.Ex;
import org.denom.crypt.ec.ECCurve;
import org.denom.crypt.hash.IHash;
import org.denom.format.ASN1OID;
import org.denom.format.BerTLV;
import org.denom.format.BerTLVList;

public class ECAlg {
    private final ECCurve curve;
    private final BigInteger N;
    private final BigInteger H;
    private final BigInteger HInv;
    private final int nSize;
    private ECCurve.ECPoint Q = null;
    private BigInteger D = null;
    private Random algRandom;
    private BigInteger fixedK = null;
    private static final Binary OID_EC_KEY = ASN1OID.toBin("1.2.840.10045.2.1");
    private final Binary oidEcParams;

    public ECAlg(ECCurve curve) {
        this(curve, new SecureRandom());
    }

    public ECAlg(ECCurve curve, Random algRandom) {
        this.curve = curve;
        this.oidEcParams = !curve.getOid().isEmpty() ? ASN1OID.toBin(curve.getOid()) : Binary.Bin();
        this.N = curve.getOrder();
        this.H = curve.getCofactor();
        this.HInv = this.H.modInverse(this.N);
        this.nSize = (curve.getFieldSize() + 7) / 8;
        this.algRandom = algRandom;
    }

    public ECAlg clone() {
        return new ECAlg(this.curve, this.algRandom);
    }

    public ECCurve getCurve() {
        return this.curve;
    }

    private void checkPublic() {
        if (this.Q != null) {
            return;
        }
        Ex.MUST(this.D != null, "Private key for ECDSA not set");
        this.Q = this.curve.GMul(this.D).normalize();
    }

    public ECAlg setPublic(Binary publicKey) {
        this.Q = this.curve.decodePoint(publicKey);
        return this;
    }

    public ECAlg setPublicX509(Binary tlv) {
        Binary key = new BerTLVList((Binary)tlv).find((String)"30/03").value;
        key = key.last(key.size() - 1);
        this.Q = this.curve.decodePoint(key);
        return this;
    }

    public Binary getPublic() {
        return this.getPublic(true);
    }

    public Binary getPublic(boolean compressed) {
        this.checkPublic();
        return this.Q.getEncoded(compressed);
    }

    public Binary getPublicX509() {
        Binary key = this.getPublic(false);
        Binary res = BerTLV.Tlv(48, "" + BerTLV.Tlv(48, "" + BerTLV.Tlv(6, OID_EC_KEY) + BerTLV.Tlv(6, this.oidEcParams)) + BerTLV.Tlv(3, "00" + key));
        return res;
    }

    public ECAlg setPrivate(Binary privateKey) {
        Ex.MUST(privateKey.size() == this.nSize, "Wrong PrivateKey size for ECDSA");
        this.D = new BigInteger(1, privateKey.getBytes());
        return this;
    }

    public ECAlg setPrivatePKCS8(Binary tlv) {
        BerTLVList l = new BerTLVList(tlv);
        l.assign(l.find((String)"30/04").value);
        Binary d = l.find((String)"30/04").value;
        if (d.size() < this.nSize) {
            Binary b = Binary.Bin(this.nSize);
            b.set(this.nSize - d.size(), d, 0, d.size());
            d = b;
        }
        this.setPrivate(d);
        return this;
    }

    public Binary getPrivate() {
        Ex.MUST(this.D != null, "Private key for ECDSA not set");
        return ECCurve.BigInt2Bin(this.nSize, this.D);
    }

    public Binary getPrivatePKCS8() {
        Ex.MUST(this.D != null, "Private key for ECDSA not set");
        Binary key = ECCurve.BigInt2Bin(this.nSize, this.D);
        Binary keyInfo = BerTLV.Tlv(48, "02 01 01" + BerTLV.Tlv(4, key));
        Binary res = BerTLV.Tlv(48, "02 01 00" + BerTLV.Tlv(48, "" + BerTLV.Tlv(6, OID_EC_KEY) + BerTLV.Tlv(6, this.oidEcParams)) + BerTLV.Tlv(4, keyInfo));
        return res;
    }

    public ECAlg generateKeyPair() {
        int nBitLength = this.N.bitLength();
        int minWeight = nBitLength >>> 2;
        do {
            this.D = new BigInteger(nBitLength, this.algRandom);
        } while (this.D.compareTo(BigInteger.valueOf(2L)) <= 0 || this.D.compareTo(this.N) >= 0 || ECCurve.getNafWeight(this.D) < minWeight);
        this.Q = this.curve.GMul(this.D).normalize();
        return this;
    }

    public void setFixedK(Binary k) {
        this.fixedK = new BigInteger(1, k.getBytes());
    }

    private BigInteger generateK() {
        BigInteger k;
        if (this.fixedK != null) {
            return this.fixedK;
        }
        int nBitLen = this.N.bitLength();
        while ((k = new BigInteger(nBitLen, this.algRandom)).equals(BigInteger.ZERO) || k.compareTo(this.N) >= 0) {
        }
        return k;
    }

    protected BigInteger calculateE(BigInteger n, byte[] message) {
        int log2n = n.bitLength();
        int messageBitLength = message.length * 8;
        BigInteger e = new BigInteger(1, message);
        if (log2n < messageBitLength) {
            e = e.shiftRight(messageBitLength - log2n);
        }
        return e;
    }

    public Binary signECDSA(Binary messageHash) {
        BigInteger s;
        BigInteger r;
        Ex.MUST(this.D != null, "Private key for ECAlg not set");
        BigInteger e = this.calculateE(this.N, messageHash.getBytes());
        do {
            BigInteger k = this.generateK();
            ECCurve.ECPoint p = this.curve.GMul(k).normalize();
            r = p.getAffineXCoord().toBigInteger().mod(this.N);
            s = k.modInverse(this.N).multiply(e.add(this.D.multiply(r))).mod(this.N);
        } while (r.equals(BigInteger.ZERO) || s.equals(BigInteger.ZERO));
        return Binary.Bin(ECCurve.BigInt2Bin(this.nSize, r), ECCurve.BigInt2Bin(this.nSize, s));
    }

    public Binary signECDSAStd(Binary messageHash) {
        BigInteger s;
        BigInteger r;
        Ex.MUST(this.D != null, "Private key for ECAlg not set");
        BigInteger e = this.calculateE(this.N, messageHash.getBytes());
        do {
            BigInteger k = this.generateK();
            ECCurve.ECPoint p = this.curve.GMul(k).normalize();
            r = p.getAffineXCoord().toBigInteger().mod(this.N);
            s = k.modInverse(this.N).multiply(e.add(this.D.multiply(r))).mod(this.N);
        } while (r.equals(BigInteger.ZERO) || s.equals(BigInteger.ZERO));
        return BerTLV.Tlv(48, "" + BerTLV.Tlv(2, Binary.Bin(r.toByteArray())) + BerTLV.Tlv(2, Binary.Bin(s.toByteArray())));
    }

    private boolean verifyECDSAImpl(Binary hash, BigInteger r, BigInteger s) {
        this.checkPublic();
        BigInteger e = this.calculateE(this.N, hash.getBytes());
        if (r.compareTo(BigInteger.ONE) < 0 || r.compareTo(this.N) >= 0) {
            return false;
        }
        if (s.compareTo(BigInteger.ONE) < 0 || s.compareTo(this.N) >= 0) {
            return false;
        }
        BigInteger c = s.modInverse(this.N);
        BigInteger u1 = e.multiply(c).mod(this.N);
        BigInteger u2 = r.multiply(c).mod(this.N);
        ECCurve.ECPoint point = this.curve.sumOfTwoMultiplies(this.curve.getG(), u1, this.Q, u2);
        if (point.isInfinity()) {
            return false;
        }
        BigInteger v = point.normalize().getAffineXCoord().toBigInteger().mod(this.N);
        return v.equals(r);
    }

    public boolean verifyECDSA(Binary hash, Binary sign) {
        BigInteger r = new BigInteger(1, sign.first(sign.size() / 2).getBytes());
        BigInteger s = new BigInteger(1, sign.last(sign.size() / 2).getBytes());
        return this.verifyECDSAImpl(hash, r, s);
    }

    public boolean verifyECDSAStd(Binary hash, Binary sign) {
        BerTLV tlv = new BerTLV(sign);
        BigInteger r = new BigInteger(tlv.find((int)2, (int)1).value.getBytes());
        BigInteger s = new BigInteger(tlv.find((int)2, (int)2).value.getBytes());
        return this.verifyECDSAImpl(hash, r, s);
    }

    public Binary signECSDSA(Binary message, IHash hashAlg) {
        BigInteger s;
        BigInteger r;
        Ex.MUST(this.D != null, "Private key for ECAlg not set");
        do {
            BigInteger k = this.generateK();
            ECCurve.ECPoint q = this.curve.GMul(k).normalize();
            Binary qx = ECCurve.BigInt2Bin(this.nSize, q.getAffineXCoord().toBigInteger());
            Binary qy = ECCurve.BigInt2Bin(this.nSize, q.getAffineYCoord().toBigInteger());
            Binary hash = hashAlg.calc(Binary.Bin(qx, qy, message));
            r = this.calculateE(this.N, hash.getBytes());
            s = r.multiply(this.D).add(k).mod(this.N);
        } while (r.equals(BigInteger.ZERO) || s.equals(BigInteger.ZERO));
        return Binary.Bin(ECCurve.BigInt2Bin(this.nSize, r), ECCurve.BigInt2Bin(this.nSize, s));
    }

    public boolean verifyECSDSA(Binary message, IHash hashAlg, Binary sign) {
        ECCurve.ECPoint rP;
        this.checkPublic();
        BigInteger r = new BigInteger(1, sign.first(sign.size() / 2).getBytes());
        BigInteger s = new BigInteger(1, sign.last(sign.size() / 2).getBytes());
        if (r.compareTo(BigInteger.ONE) < 0 || r.compareTo(this.N) >= 0) {
            return false;
        }
        if (s.compareTo(BigInteger.ONE) < 0 || s.compareTo(this.N) >= 0) {
            return false;
        }
        ECCurve.ECPoint sG = this.curve.GMul(s);
        ECCurve.ECPoint q = sG.subtract(rP = this.Q.multiply(r)).normalize();
        if (q.isInfinity()) {
            return false;
        }
        Binary qx = ECCurve.BigInt2Bin(this.nSize, q.getAffineXCoord().toBigInteger());
        Binary qy = ECCurve.BigInt2Bin(this.nSize, q.getAffineYCoord().toBigInteger());
        Binary hash = hashAlg.calc(Binary.Bin(qx, qy, message));
        BigInteger v = this.calculateE(this.N, hash.getBytes());
        return v.equals(r);
    }

    public Binary signECSDSA_X(Binary message, IHash hashAlg) {
        Binary R;
        BigInteger s;
        BigInteger r;
        Ex.MUST(this.D != null, "Private key for ECAlg not set");
        do {
            BigInteger k = this.generateK();
            ECCurve.ECPoint q = this.curve.GMul(k).normalize();
            Binary x1 = ECCurve.BigInt2Bin(this.nSize, q.getAffineXCoord().toBigInteger());
            R = hashAlg.calc(Binary.Bin(x1, message));
            r = new BigInteger(1, R.getBytes()).mod(this.N);
            s = r.multiply(this.D).add(k).mod(this.N);
        } while (r.equals(BigInteger.ZERO) || s.equals(BigInteger.ZERO));
        return Binary.Bin(R, ECCurve.BigInt2Bin(this.nSize, s));
    }

    public boolean verifyECSDSA_X(Binary message, IHash hashAlg, Binary sign) {
        ECCurve.ECPoint rQ;
        this.checkPublic();
        int hashSize = hashAlg.size();
        if (sign.size() != hashSize + this.nSize) {
            return false;
        }
        Binary R = sign.first(hashAlg.size());
        BigInteger r = new BigInteger(1, R.getBytes()).mod(this.N);
        BigInteger s = new BigInteger(1, sign.last(this.nSize).getBytes());
        if (r.compareTo(BigInteger.ONE) < 0 || s.compareTo(BigInteger.ONE) < 0 || s.compareTo(this.N) >= 0) {
            return false;
        }
        ECCurve.ECPoint sG = this.curve.GMul(s);
        ECCurve.ECPoint q = sG.subtract(rQ = this.Q.multiply(r)).normalize();
        if (q.isInfinity()) {
            return false;
        }
        Binary x2 = ECCurve.BigInt2Bin(this.nSize, q.getAffineXCoord().toBigInteger());
        Binary R2 = hashAlg.calc(Binary.Bin(x2, message));
        return R2.equals(R);
    }

    public Binary signECGDSA(Binary messageHash) {
        BigInteger s;
        BigInteger r;
        Ex.MUST(this.D != null, "Private key for ECAlg not set");
        BigInteger e = this.calculateE(this.N, messageHash.getBytes());
        do {
            BigInteger k = this.generateK();
            ECCurve.ECPoint q = this.curve.GMul(k).normalize();
            r = q.getAffineXCoord().toBigInteger().mod(this.N);
            s = k.multiply(r).subtract(e).multiply(this.D).mod(this.N);
        } while (r.equals(BigInteger.ZERO) || s.equals(BigInteger.ZERO));
        return Binary.Bin(ECCurve.BigInt2Bin(this.nSize, r), ECCurve.BigInt2Bin(this.nSize, s));
    }

    public boolean verifyECGDSA(Binary hash, Binary sign) {
        this.checkPublic();
        BigInteger r = new BigInteger(1, sign.first(sign.size() / 2).getBytes());
        BigInteger s = new BigInteger(1, sign.last(sign.size() / 2).getBytes());
        BigInteger h = this.calculateE(this.N, hash.getBytes());
        if (r.compareTo(BigInteger.ONE) < 0 || r.compareTo(this.N) >= 0) {
            return false;
        }
        if (s.compareTo(BigInteger.ONE) < 0 || s.compareTo(this.N) >= 0) {
            return false;
        }
        BigInteger r1 = r.modInverse(this.N);
        BigInteger u1 = r1.multiply(h).mod(this.N);
        BigInteger u2 = r1.multiply(s).mod(this.N);
        ECCurve.ECPoint point = this.curve.sumOfTwoMultiplies(this.curve.getG(), u1, this.Q, u2);
        if (point.isInfinity()) {
            return false;
        }
        BigInteger v = point.normalize().getAffineXCoord().toBigInteger().mod(this.N);
        return v.equals(r);
    }

    public Binary signECGOST(Binary hash) {
        BigInteger s;
        BigInteger r;
        Ex.MUST(this.D != null, "Private key for ECGOST not set");
        BigInteger e = new BigInteger(1, hash.getBytes());
        do {
            BigInteger k = this.generateK();
            ECCurve.ECPoint p = this.curve.GMul(k).normalize();
            r = p.getAffineXCoord().toBigInteger().mod(this.N);
            s = k.multiply(e).add(this.D.multiply(r)).mod(this.N);
        } while (r.equals(BigInteger.ZERO) || s.equals(BigInteger.ZERO));
        return Binary.Bin(ECCurve.BigInt2Bin(this.nSize, r), ECCurve.BigInt2Bin(this.nSize, s));
    }

    public boolean verifyECGOST(Binary hash, Binary sign) {
        this.checkPublic();
        BigInteger r = new BigInteger(1, sign.first(sign.size() / 2).getBytes());
        BigInteger s = new BigInteger(1, sign.last(sign.size() / 2).getBytes());
        BigInteger e = new BigInteger(1, hash.getBytes());
        if (r.compareTo(BigInteger.ONE) < 0 || r.compareTo(this.N) >= 0) {
            return false;
        }
        if (s.compareTo(BigInteger.ONE) < 0 || s.compareTo(this.N) >= 0) {
            return false;
        }
        BigInteger v = e.modInverse(this.N);
        BigInteger z1 = s.multiply(v).mod(this.N);
        BigInteger z2 = this.N.subtract(r).multiply(v).mod(this.N);
        ECCurve.ECPoint point = this.curve.sumOfTwoMultiplies(this.curve.getG(), z1, this.Q, z2).normalize();
        if (point.isInfinity()) {
            return false;
        }
        BigInteger R = point.getAffineXCoord().toBigInteger().mod(this.N);
        return R.equals(r);
    }

    public ECCurve.ECPoint[] encrypt(ECCurve.ECPoint point) {
        this.checkPublic();
        BigInteger k = this.generateK();
        return new ECCurve.ECPoint[]{this.curve.GMul(k).normalize(), this.Q.multiply(k).add(this.curve.cleanPoint(point)).normalize()};
    }

    public ECCurve.ECPoint decrypt(ECCurve.ECPoint[] pair) {
        Ex.MUST(this.D != null, "Private key for EC decrypt not set");
        ECCurve.ECPoint tmp = this.curve.cleanPoint(pair[0]).multiply(this.D);
        ECCurve.ECPoint res = this.curve.cleanPoint(pair[1]).subtract(tmp).normalize();
        return res;
    }

    public ECCurve.ECPoint[] transformRandom(ECCurve.ECPoint[] cipherText) {
        this.checkPublic();
        BigInteger k = this.generateK();
        return new ECCurve.ECPoint[]{this.curve.GMul(k).add(this.curve.cleanPoint(cipherText[0])).normalize(), this.Q.multiply(k).add(this.curve.cleanPoint(cipherText[1])).normalize()};
    }

    public ECCurve.ECPoint[] transform(ECCurve.ECPoint[] cipherText) {
        this.checkPublic();
        BigInteger k = this.generateK();
        return new ECCurve.ECPoint[]{this.curve.GMul(k).normalize(), this.Q.multiply(k).add(this.curve.cleanPoint(cipherText[1])).normalize()};
    }

    public Binary calcDH(Binary otherPublicQ) {
        ECCurve.ECPoint p;
        Ex.MUST(this.D != null, "Private key for ECDH not set");
        ECCurve.ECPoint q = this.curve.decodePoint(otherPublicQ);
        BigInteger d = this.D;
        if (!this.H.equals(BigInteger.ONE)) {
            d = this.HInv.multiply(d).mod(this.N);
            q = q.multiply(this.H);
        }
        Ex.MUST(!(p = q.multiply(d).normalize()).isInfinity(), "Infinity is not a valid agreement value for ECDH");
        return ECCurve.BigInt2Bin(this.nSize, p.getAffineXCoord().toBigInteger());
    }

    public Binary calcDHC(Binary otherPublicQ) {
        Ex.MUST(this.D != null, "Private key for ECDH not set");
        BigInteger hd = this.H.multiply(this.D).mod(this.N);
        ECCurve.ECPoint q = this.curve.decodePoint(otherPublicQ);
        ECCurve.ECPoint p = q.multiply(hd).normalize();
        Ex.MUST(!p.isInfinity(), "Infinity is not a valid agreement value for ECDHC");
        return ECCurve.BigInt2Bin(this.nSize, p.getAffineXCoord().toBigInteger());
    }

    public Binary calcECDHCUnified(ECAlg ephemeralPrivate, Binary otherStaticPublic, Binary otherEphemeralPublic) {
        return Binary.Bin(ephemeralPrivate.calcDHC(otherEphemeralPublic), this.calcDHC(otherStaticPublic));
    }

    public Binary calcMQVAgreement(ECAlg ephemeral, Binary otherStaticPublic, Binary otherEphemeralPublic) {
        Ex.MUST(this.D != null, "Private key for ECDH not set");
        int e = (this.N.bitLength() + 1) / 2;
        BigInteger powE = BigInteger.ONE.shiftLeft(e);
        ECCurve.ECPoint q1v = this.curve.decodePoint(otherStaticPublic);
        ECCurve.ECPoint q2v = this.curve.decodePoint(otherEphemeralPublic);
        BigInteger x = ephemeral.Q.getAffineXCoord().toBigInteger();
        BigInteger xBar = x.mod(powE);
        BigInteger Q2UBar = xBar.setBit(e);
        BigInteger s = this.D.multiply(Q2UBar).add(ephemeral.D).mod(this.N);
        BigInteger xPrime = q2v.getAffineXCoord().toBigInteger();
        BigInteger xPrimeBar = xPrime.mod(powE);
        BigInteger Q2VBar = xPrimeBar.setBit(e);
        BigInteger hs = this.H.multiply(s).mod(this.N);
        ECCurve.ECPoint p = this.curve.sumOfTwoMultiplies(q1v, Q2VBar.multiply(hs).mod(this.N), q2v, hs).normalize();
        Ex.MUST(!p.isInfinity(), "Infinity is not a valid agreement value for MQV");
        return ECCurve.BigInt2Bin(this.nSize, p.getAffineXCoord().toBigInteger());
    }
}

